/* 
 * extract.c reads data according to the locations in a catalog file,
 * and produces another data-file for the start to end dates you specify 
 * on its command line:
 * 
 *	extract <cat-file> <outputfile> jdStart jdEnd 
 * 
 * for more comment on how this works, see convert.doc
 *
 * andrew p. porter, porter@s1.gov, 1994/summer
 *
 */ 

#include <stdio.h>
#include <stdlib.h>

#include "../include/ms.h"
#include "../include/matrix.h"
#include "../include/hnav.h"

/* this next has to be less than the length of an ephemeris record; 
 * it is the step size from the end of one record 
 * into the middle of the next
 */ 
#define EPSILON 1.

#define LINE_LENGTH 128    /* input file line length */ 
#define DATA_SIZE 1300    /* how many lines to an ascii data-record */ 

void extractData();
void getRecord();
void putRecord();
void loadData(); 
void getDataFileName(); 
char *fvfShortJulianToDate(); 

main( argc, argv )
     int argc;
     char *argv[];
{
    char *usage="usage: extractdates catalogfilename outputfilename from year mo day to year mo day \n";  
    double startJD;	/* first jd of the output file */ 
    double stopJD;	/* last jd of the output file  */ 

    char catalog[80];
    char outfilename[80];
    int icomp; 	
    
    /* if we don't have right number of args, print usage and quit */ 
    if ( argc != 11 )
    {
	printf(" wrong number of arguments (%d): \n", argc );  
	printf("%s", usage );
	exit(0);
    }

    /* recklessly assume the arguments are valid ... */ 
    strcpy( catalog, argv[1] );
    strcpy( outfilename, argv[2] );
    if ( (icomp = strncmp( argv[3], "from", 4 )) == 0 )
      startJD = calendarDateToJulian( atoi(argv[4]), atoi(argv[5]), atoi(argv[6]) );
    else
    {
	printf(" from: icomp = %d, argv[4] = %s \n", icomp, argv[4] ); 
	printf("%s", usage );
	exit(0);
    }
    if ( (icomp = strncmp( argv[7], "to", 2 )) == 0 )
      stopJD = calendarDateToJulian( atoi(argv[8]), atoi(argv[9]), atoi(argv[10]) );
    else
    {
	printf(" to: icomp = %d, argv[8] = %s \n", icomp, argv[8] ); 
	printf("%s", usage );
	exit(0);
    }
    
    printf(" about to create ephemeris file %s according to catalog %s, \n",
	   outfilename, catalog );
    printf(" for dates %12.2f to %12.2f \n", startJD, stopJD ); 

    /* this actually does the work: */ 
    extractData( catalog, outfilename, startJD, stopJD );

    exit( 0 ); 	/* report success */ 

}


/* ----------------------------------------------------------------
 * extractData reads data from data-files named in a catalogfile,
 *     inclusive of the Julian dates specified, 
 *     and creates the specified output file
 * inputs
 *     catalogfile name 
 *     output file name 
 *     start JD
 *     stop JD
 * outputs
 *     none
 * side effects
 *     creates the output file and writes to it 
 * ----------------------------------------------------------------
 */ 

void
extractData( catalog, outfilename, startJD, stopJD ) 
    char catalog[80];
    char outfilename[80];
    double startJD;	/* first jd of the output file */ 
    double stopJD;	/* last jd of the output file  */ 
{
    FILE *fo; 

    double jd;
    double jdStart;	/* first jd of a record */ 
    double jdEnd; 	/* last  jd of a record */ 

    char data[DATA_SIZE][LINE_LENGTH];
    int recordLength;	/* how many lines of data were put into data[][] */ 
    int i,k; 

    for( i=0; i<DATA_SIZE; ++i)
      strcpy( data[i], "empty line" ); 

    fo = fopen( outfilename, "w" );
    if (fo == NULL)
    {
	printf(" could not open %s \n", outfilename );
	exit(0);
    }

    for( jd = startJD; ; )
    {
	printf(" about to get record for Julian date %12.2f, %s \n", 
	       jd, fvfShortJulianToDate( jd )   ); 
	getRecord( catalog, data, &jdStart, &jdEnd, &recordLength, jd );
	putRecord( fo, data, recordLength );
	if ( jdEnd > stopJD)
	  break;
	jd = jdEnd + EPSILON;
    }

    fclose( fo ); 

}



/* ----------------------------------------------------------------
 * getRecord manages the call to loadData
 * and passes on all its arguments but the first to loadData
 * note the first arg tells indirectly where to get the data
 * args 2-5 specify the data to be found, 
 * the last determines which record to get
 * 
 * inputs
 *     catalogfile, tells where to find the datefile
 *     address of an array of character strings to put the date into
 *     jd-start of the data that is found 
 *     jd-end   of the data that is found 
 *     the record length, i. e., number of lines found
 *     the jd that the record is supposed to include 
 * outputs
 *     none
 * side effects
 *     writes to the array data[][],
 *     sets xjdStart, xjdEnd, xcrecordLength
 * ----------------------------------------------------------------
 */ 

void 
getRecord( catalog, data, xjdStart, xjdEnd, xrecordLength, jd )
     char *catalog; 	/* the catalog file that tells where the data files are */ 
     char data[DATA_SIZE][LINE_LENGTH];	/* where to put the data */ 
     double *xjdStart;	/* start jd of the record to be read in */
     double *xjdEnd;	/* end jd of the record to be read in */ 
     int *xrecordLength; /* how many lines of data were put into data[][] */ 
     double jd;		/* jd to be included in the record sought */ 
{
    char inputFileName[32];	/* ephemeris data file with desired data in it */ 
    double jdStart;
    double jdEnd; 
    int recordLength; 
    

    getDataFileName( catalog, jd, inputFileName ); 

    loadData( inputFileName, data, &jdStart, &jdEnd, &recordLength, jd );

    *xjdStart = jdStart;
    *xjdEnd   = jdEnd; 
    *xrecordLength = recordLength; 
}


/* ----------------------------------------------------------------
 * loadData does the work of finding the data
 * getRecord has looked up the data-file in the catalog file
 * args 2-5 specify the data to be found, 
 * the last determines which record to get
 * 
 * inputs
 *     data-file to search for the record sought 
 *     address of an array of character strings to put the date into
 *     jd-start of the data that is found 
 *     jd-end   of the data that is found 
 *     the record length, i. e., number of lines found
 *     the jd that the record is supposed to include 
 * outputs
 *     none
 * side effects
 *     writes to the array data[][],
 *     sets xjdStart, xjdEnd, xcrecordLength
 * ----------------------------------------------------------------
 */ 

void 
loadData( inputFileName, data, xjdStart, xjdEnd, xrecordLength, jd )
     char *inputFileName;
     char data[DATA_SIZE][LINE_LENGTH];
     double *xjdStart;
     double *xjdEnd; 
     int *xrecordLength; 
     double jd;
{
    static FILE *fi;
    static char infile[80]; 
    char buf[120];
    int count; 		/* return value from sscanf */ 
    int recordNumber;	/* record number of the dat record */ 
    double jdStart, jdEnd; 
    int i;

    if ( strcmp( inputFileName, infile ) != 0 )
    {
	strcpy( infile, inputFileName ); 
	if (fi != NULL)
	  fclose( fi ); 
	fi = fopen( infile, "r" ); 
    }

    if ( fi == NULL)
    {
	printf(" something is wrong; \n" );
	printf(" we should have a file open, in loadData() \n");
	exit(0);
    }

    /* pass over data ahead of the record we want */ 
    while( 1 ) 
    {
	fgets( buf, LINE_LENGTH, fi );
	count = sscanf( buf, "record %le to %le", &jdStart, &jdEnd );
	if (count != 2)
	  continue; 
	strcpy( data[0], buf );

	if ( jd >= jdStart && jd <= jdEnd )
	  break; 	/* this is the record we want; break out */ 
    }


    /* copy the record we want, line by line, into data[][] */ 
    for( i=1; i<DATA_SIZE; ++i) 
    {
	fgets( buf, LINE_LENGTH, fi );
	strcpy( data[i], buf ); 

	count = sscanf( buf, "end of record %le to %le", &jdStart, &jdEnd );
	if (count == 2)	/* end of record line; you're done */ 
	{
	    break;
	}

    }

    *xjdStart = jdStart;
    *xjdEnd   = jdEnd; 
    *xrecordLength = i+1; 
    
}



void 
getDataFileName( catalog, jd, inputFileName )
     char *catalog;		/* where to look, the catalog file */ 
     double jd;			/* the jd for which a file is sought */ 
     char *inputFileName;	/* where to put the name of the file sought */ 
{
    char buf[120]; 
    static FILE *fi;
    static char catfile[80]; 
    char *line; 
    int count; 
    double jdStart;
    double jdEnd;
    char dataFileName[80]; 

    if ( strcmp( catalog, catfile ) != 0 )
    {
	strcpy( catfile, catalog ); 
	if (fi != NULL)
	  fclose( fi ); 
	fi = fopen( catfile, "r" ); 
    }

    if ( fi == NULL)
    {
	printf(" something is wrong; \n" ); 
	printf(" we should have a file open, in getDataFileName() \n");
	exit(0);
    }

    rewind( fi ); 

    while( 1 )
    {
	line = fgets( buf, 120, fi );
	if ( line == NULL) break; 

	count = sscanf( buf, "start: %le end: %lf %s", 
		       &jdStart, &jdEnd, &dataFileName );
	if ( count != 3 )
	{
	    printf(" something odd in catalog file %s \n", catfile );
	    exit(0);
	}

	if ( jd < jdStart || jd > jdEnd )
	  continue;	/* keep looking */ 
	else
	{
	    strcpy( inputFileName, dataFileName );
	    return;
	}
	
    }

    printf(" could not find data file for jd %12.2f in catfile %s \n",
	   jd, catfile );

}


void       
putRecord( fo, data, recordLength )
     FILE *fo;
     char data[DATA_SIZE][LINE_LENGTH];
     int recordLength;
{
    int i;

    for( i=0; i<recordLength; ++i )
    {
	fprintf( fo, "%s", data[i] );
	fflush( fo ); 
    }

}


/* ----------------------------------------------------------------
 * fvfShortJulianToDate is Fliegel and Van Flandern's algorithm
 * 
 * inputs
 *	julian date
 * outputs
 *	ptr to gregorian calendar date string
 * side effects
 *	sets calendar date integers pointed to on input
 * method
 *	convert JD to unix-time, and unix-time to Gregorian calendar date 
 * cautions
 *	use the answer before you loose it . . . it is in static storage here
 * Source 
 *	Henry F. Fliegel, Thomas C. Van Flandern, "Machine Algorithm 
 *	for Processing Calendar Dates," 
 *              [Commmunications?] of the ACM, 11#10 (1968/10) ?
 *	modulo changes in NOTES below, 
 *		this is code from ~/nav/calendar/calendar.c, 94/08/15
 *		here in order to compile "stand-alone", w/o ephlib library 
 * NOTES
 *	output string has been truncated to date, w/o time of day 
 * ----------------------------------------------------------------
 */

char * 
fvfShortJulianToDate( jd )
     double  jd;	/* floating-point Julian date  */
{
    double jdnoon;	/* jd at noon on the day in question */ 
    int L,N;
    int year;
    int month; 	/* month = 1-12 */ 
    int day; 	/* a presume 1 to 28-31 */ 
    static char nice[100];	/* to hold the answer */
    float fraction; 	/* fraction of day */ 
    int hour, minute, second;		/* time of day in integers */ 
    float fhour, fminute, fsecond;	/* time of day in floats */ 

    jdnoon = (double) (int) ( jd + 0.5 ); 
    fraction = jd - (jdnoon - 0.5 );

    /* Fliegel and Van Flandern's algorithm: */ 
    L = jdnoon + 68569;
    N = 4 * L / 146097; 
    L = L - (146097*N + 3) / 4; 

    year = 4000 * (L+1) / 1461001; 
    L = L - 1461 * year / 4  +  31; 
    month = 80 * L / 2447; 
    day = L - 2447 * month / 80;
    L = month / 11;
    month = month + 2 - 12 * L;
    year = 100 * ( N - 49)  +  year + L;
    /* end of F & vF's algorithm */ 

    fhour = fraction * 24.;
    hour = (int) fhour;
    fminute = (fhour - hour) * 60.;
    minute = (int) fminute; 
    fsecond = (fminute - minute) * 60.; 
    second = (int) fsecond; 

    /* now format the time in struct t to string nice: */
    (void) sprintf( nice, "%d/%2d/%2d", year, 
		  month, day ); 

    return nice;	/* for now . . . better later */

}

