
/* ----------------------------------------------------------------
 * translates odiv ascii format ephemeris file to odiv binary format
 * 		by a. porter, 89/01/09 ff., major revisions in 1994 
 *
 * ("odiv" refers to a format developed in O Division 
 *	at the Lawrence Livermore National Laboratory;
 * 	it is the same ascii chebyshev polynomial coefficients
 *	as the JPL de-245 ephemeris,
 *	but one number per line instead of three,
 *	with header data removed to another file,
 *	and with blank lines, comments to identify the body and dates 
 *	in question, and record-ID lines inserted.  
 * ----------------------------------------------------------------
 */


#include <stdio.h>
#include <ctype.h>
#include <math.h>
#include <string.h>


#include "../include/ms.h" 
#include "../include/matrix.h"
#include "../include/hnav.h" 

extern double normalize();

/* these are used only in this file: */ 
void exitIfNotODivFormat();
int isODivFormat();
void exitIfNotODivHeader();
int isODivHeader();



/* ----------------------------------------------------------------
 * translate odiv ascii ephemeris file to binary: 
 * ----------------------------------------------------------------
 */

main( argc, argv )
     int argc;		/* argument count; i. e., count of words on the command line  */
     char *argv[];	/* words on the command line */
{

    translateAsciiToBinary( argc, argv ); 

}


/* ----------------------------------------------------------------
 * odiv ascii to binary 
 * ----------------------------------------------------------------
 */ 


translateAsciiToBinary( argc, argv )
     int argc;		/* argument count; i. e., count of words on the command line  */
     char *argv[];	/* words on the command line */
{
    char *input_file;	/* file name */
    char *header_file;	/* the header file */ 
    char output_file[80];	/* file name */
    char tell_file[80];  /* tell-file name */ 

    /* translate from odiv ascii to binary format: */ 
    if (argc <= 2) 
    {
	printf(" usage: asctobin header-file data-file-to-be-translated  \n");
	exit(0);
    }

    /* my experience is that if you rely on an environmental variable
     * to specify the header-file, it will often be the wrong one,
     * and unfortunately, the resulting binary works -- too much of the time,
     * but it does not give correct answers
     * If you want the default header-file, uncomment call to getHeaderName(),
     * and comment out the next line, referencing argv[1] for header_file: 
     */ 
    /* header_file = getHeaderName();  */ 
    header_file = argv[1]; 

    input_file = argv[2];

    exitIfNotODivFormat( input_file ); /* complain and exit if it's not OD format */ 
    exitIfNotODivHeader( header_file );  /* the same for the header file */ 

    strcpy( tell_file, input_file ); 
    strcat( tell_file, ".tel" ); 

    strcpy( output_file, input_file ); 
    strcat( output_file, ".bin" ); 

    printf(" header file = %s \n", header_file );

    asciiToBinary( header_file, input_file, output_file, tell_file ); 

}
    

/* ----------------------------------------------------------------
 * asciiToBinary translates from odiv format ascii to binary 
 * inputs
 *	a header-file name
 *	input file name: the ascii file to read
 *	output file name: the binary file to write
 *	the "tell" file name, into which results of ftell() go,
 *		for later use by fseek() 
 * ouputs
 *	1 for success 
 * side effects
 *	creats and writes to output_file and tell_file 
 * ----------------------------------------------------------------
 */ 


int 
asciiToBinary( header_file, input_file, output_file, tell_file )
     char *header_file; 
     char *input_file;
     char *output_file;
     char *tell_file;  /* tell-file name */ 
{
    int recordSize;
    TapeHeap *heap;
    TapeHeap *heap1; 
    TapeHeap *heap2; 
    TapeHeap heapbuf;
    TapeHeap heap1buf; 
    TapeHeap heap2buf; 
    FILE *fh;		/* header file */ 
    FILE *fi;		/* input file */ 
    FILE *fo;		/* output binary file */ 
    FILE *ft;		/* output record of ftell results */ 
    int success; 
    double jdStart, jdEnd, jd; 
    long int where;   	/* result of ftell on fo */
    int sizewrite;	/* size of block to write */ 
    int i, j; 
    double starts[TellTableSize];	/* array of start dates */ 
    double ends[TellTableSize];		/* array of end dates */ 
    long int tells[TellTableSize]; 	/* array of ftell results, for fseek */ 
    char buf[129]; 

    fi = fopen( input_file,  "r");
    if (fi == NULL )
    {
	printf(" could not open input file %s \n", input_file );
	exit(0);
    }

    /* ----------------------------------------------------------------
     * now read the input file . . . 
     * ----------------------------------------------------------------
     */

    heap = &heapbuf; 
    success = fileToEmptyHeap( header_file, heap ); 
    if (success == 0)		/* some of these files are bad */ 
    {
	printf(" could not get a TapeHeap from file %s \n", header_file ); 
	exit(0); 
    }

    fo	 = getfile( output_file, "wb" ); 
    ft = getfile( tell_file, "w" );
    
    recordSize = getRecordLength( &heap->groupSeven ) ; 
    printf(" recordSize = %d  before fwrite() to output file \n", recordSize );

    for( i=0; i<10000; ++i)
    {
	success = readRecord( fi, heap ); 
	if (success == 0) break; 
	jdStart = heap->jdStart;
	jdEnd = heap->jdEnd; 
	printf(" jdStart %12.2f jdEnd %12.2f \n", jdStart, jdEnd ); 

	where = ftell( fo ); 
	fwrite( heap->data, sizeof( double ), recordSize, fo );
	fprintf( ft, "%d jdStart %12.2f jdEnd %12.2f %lx \n", i, jdStart, jdEnd, where ); 
	fflush( ft ); 
    }

    fflush( fo );     fclose( fo );
    fflush( ft );     fclose( ft ); 

    /* ---------------------------------------------------------------- */ 
    /* now read back the binary and ascii files together, and compare:  */ 

    fclose( fi ); 
    fi = fopen( input_file,  "r");
    fo = getfile( output_file, "rb" ); 
/*
    printf(" re-opening  input file %s \n",  input_file ); 
    printf(" re-opening output file %s \n", output_file ); 
*/
    heap1 = &heap1buf; 
    success = fileToEmptyHeap( header_file, heap1 ); 
    heap2 = &heap2buf; 
    success = fileToEmptyHeap( header_file, heap2 ); 

    for( i=0; i<recordSize; ++i) 
      heap2->data[i] = 0.; 


    printf(" reading the tell-file \n"); 
    getTellInfo( tell_file, starts, ends, tells ); 

    printf(" done reading the tell-file: \n"); 
    for( i=0; i<TellTableSize; ++i)
    {
	printf(" %d jdStart %12.2f jdEnd %12.2f tell %lx \n", 
	       i, starts[i], ends[i], tells[i] ); 
	if (starts[i] == ends[i]) break; 
    }

    /* with the call to getfile put here (instead of above), the later code works;
     *  put above only, the code following doesn't read correctly from fo: 
     */ 
    fclose( fo );
    fo = getfile( output_file, "rb" ); 

    printf(" re-read the data and check it: \n"); 
    for( i=0; i<10000; ++i)
    {
	printf(" i = %d \n", i ); 
	success = readRecord( fi, heap1 ); 
	if (success == 0) 
	{
	    printf(" readRecord returned %d \n", success ); 
	    break; 
	}
	jdStart = heap1->jdStart;
	jdEnd = heap1->jdEnd; 
	printf(" ascii  jdStart %12.2f jdEnd %12.2f \n", jdStart, jdEnd ); 

	jd = (jdStart + jdEnd) * 0.5; 
	for( j=0; j<TellTableSize; ++j)
	{
	    if (jd > starts[j] && jd < ends[j] ) break;
	}
	/* printf(" fseek to tells[%d] = %x \n", j, tells[j] );  */

	success = fseek( fo, tells[j], 0 ); 
	/* printf(" fseek returned %d ( returns 0 on success, non-zero --> fail ) \n", 
	   success ); */

	success = fread( &(heap2->data[0]), sizeof( double ), recordSize, fo ); 
	/* printf(" fread returns %d, recordSize sought = %d \n", 
	   success, recordSize );  */

	if (success < recordSize) 
	{
	    printf(" ferror returns %d \n", ferror( fo ) ); 
	    clearerr( fo ); 
	    printf(" recordsize error ... quit checking \n"); 
	    printf("   i, heap2->data[i], heap1->data[i]  \n" );
	    for( i=0; i<success; ++i)
	    {
		printf(" i %d   %e   %e \n", i, heap2->data[i], heap1->data[i] );
	    }
	    break;  
	}
	heap2->jdStart = heap2->data[0];
	heap2->jdEnd   = heap2->data[1]; 
	jdStart = heap2->jdStart;
	jdEnd = heap2->jdEnd; 
	printf( " binary jdStart %12.2f jdEnd %12.2f %lx \n", 
	       jdStart, jdEnd, tells[j] );

	success = 1; 
	printf("\n check: "); 
	for(j=0; j<recordSize; ++j)
	{
	    if ( heap1->data[j] != heap2->data[j] )
	    {
		printf("      bad %d %e %e %e \n", 
		       j, heap1->data[j], heap2->data[j], heap1->data[j] - heap2->data[j] );  
	    }
	}
	printf(" i %d, success = %d (1=yes, ok; 0=no, errors) \n", i, success ); 


    }


    return 1; 

}


/* ----------------------------------------------------------------
 * exitIfNotODivFormat tests to see if the file named in its argument
 *      is an O-Div-ascii format file;
 *      if it is not, asctobin will choke on it, 
 *      so complain and quit ...
 * inputs
 *      a filename to be checked 
 * outputs 
 *      none
 * side effects
 *      exits if the file is not in O-Div format,
 *      does nothing and returns if it is 
 * ----------------------------------------------------------------
 */ 

void
exitIfNotODivFormat( filename )
     char *filename;
{
  FILE *fi;
  char linebuf[120];

  fi = fopen( filename, "r" );

  if ( !isODivFormat( fi ) )
    {
      printf(" your ascii input file is not in O-Div format; \n");
      printf(" you must convert it to O-Division format before converting to binary \n");
      printf(" exiting now; type <convert> to see convert usage \n"); 

      exit(1);

    }

  return; 

}



/* ----------------------------------------------------------------
 * isODivFormat, given a file-descriptor, inspects to see if it is in O-Div format
 * inputs
 *      file descriptor
 * outputs
 *      1 for yes, 0 for no
 * side effects
 *      none 
 * method
 *      looks for a line that contains the word "record" 
 *      (yes, its crude ... ) 
 * ----------------------------------------------------------------
 */ 


int
isODivFormat( fi )
     FILE *fi;
{
  char linebuf[120];
  char *line;
  int i;        /* loop counter */ 

  for( i=0; i<10; ++i)   /* search the first ten lines */ 
    {
      line = fgets( linebuf, 120, fi ); 
  
      if (strstr( line, "record" ) != NULL)
	return 1;       /* found a match, it is O-Div format: */ 
    }

  /* did not find a match, it is not in O-Div format: */ 
  return 0; 

}



/* ----------------------------------------------------------------
 * exitIfNotODivHeader tests to see if the file named in its argument
 *      is an O-Div format header file;
 *      if it is not, asctobin will choke on it, 
 *      so complain and quit ...
 * inputs
 *      a filename to be checked 
 * outputs 
 *      none
 * side effects
 *      exits if the file is not in O-Div format,
 *      does nothing and returns if it is 
 * ----------------------------------------------------------------
 */ 

void
exitIfNotODivHeader( filename )
     char *filename;
{
  FILE *fi;
  char linebuf[120];

  fi = fopen( filename, "r" );

  if ( !isODivFormat( fi ) )
    {
      printf(" your ascii header file is not in O-Div format; \n");
      printf(" you must use an O-Div header file as input to asctobin \n");
      printf(" exiting now \n"); 

      exit(1);

    }

  return; 

}



/* ----------------------------------------------------------------
 * isODivHeader, given a file-descriptor, 
 *      inspects to see if it is an O-Div format header file 
 * inputs
 *      file descriptor
 * outputs
 *      1 for yes, 0 for no
 * side effects
 *      none 
 * method
 *      looks for a line that contains the word "earliestDate" 
 *      (yes, its crude ... ) 
 * ----------------------------------------------------------------
 */ 


int
isODivHeader( fi )
     FILE *fi;
{
  char linebuf[120];
  char *line;
  int i;        /* loop counter */ 

  for( i=0; i<10; ++i)   /* search the first ten lines */ 
    {
      line = fgets( linebuf, 120, fi ); 
  
      if (strstr( line, "earliestDate" ) != NULL)
	return 1;       /* found a match, it is O-Div format: */ 
    }

  /* did not find a match, it is not in O-Div format: */ 
  return 0; 

}




