/* 
 * fastplan.c prints planetary positions 
 * calculated from Keplerian orbits in the _Explanatory Supplement_, p. 316
 * with no apparent place corrections
 * Andrew P. Porter, 1994
 */
 

#include <stdio.h>
#include <stdlib.h> 
#include <math.h>
#include <time.h>
 
 
#include "../include/ms.h"
#include "../include/matrix.h"
#include "../include/hnav.h"
 

 
extern Orbit *planet[];
extern char  *planetName[];
 
/* local subroutines: */  
void fastPlanets();
double xjdnow(); 
void usage();  
double getJD(); 

void tabulateHours();
void tabulateDegrees();



 
/* ----------------------------------------------------------------
 * main reads command-line requests and passes them on 
 * to data-reporting routines 
 * NOTE -- the format for requests is different from planets.c
 * ----------------------------------------------------------------
 */
 
main( argc, argv )
     int argc;
     char *argv[];
{
    double jd; 
 
    /* -------------------------------- */ 
    jd = getJD( argc, argv ); 
    
    /* -------------------------------- */ 
    /* from old code */ 
    fastPlanets( jd ); 
 
    return 0; 
 
}
 
 

/* ----------------------------------------------------------------
 * getJD gets the julian day number for which positions are desired,
 *	from the command line:
 * if there is only one argument, it is assumed to be the JD;
 * if there are three arguments, they are assumed to be a calendar date
 * if there are no arguments, return the jd of current time.
 * 
 * inputs
 *	count and pointer to command line arguments
 * outputs
 *	a double-float julian day number
 * side effects
 *	none 
 * ----------------------------------------------------------------
 */ 


double 
getJD( argc, argv )
  int argc;
  char *argv[];
{

    if ( argc == 1 )
      return xjdnow(); 

    if ( argc == 2 )
      return atof( argv[1] );

    /* this truncates the date to an integer: */ 

    if ( argc == 4 )
      return
	calendarDateToJulian(
			     atoi( argv[1] ),
			     atoi( argv[2] ),
			     atoi( argv[3] )
			     );
    else
      usage(); 	/* this exits */ 

}


 
 
 

/* ----------------------------------------------------------------
 * usage prints instructions for usage and quits
 * ----------------------------------------------------------------
 */ 


void usage()
{
    printf(" usage: \n" );
    printf(" fast [no arguments] gets you planet positions for current time/date \n");
    printf(" fast [1 argument] gets you positions for date = 1st arg, interp as Julian date \n");
    printf(" fast [3 args] gets you positions for <year, month, date> \n"); 
    printf(" note -- this is not the same syntax as planets.c \n"); 
    exit(0);
}



 

/* ----------------------------------------------------------------
 * this prints out planet positions for the jd in its argument
 * first in barycentric ecliptic, then geocentric equatorial coords 
 * ----------------------------------------------------------------
 */ 



void 
fastPlanets( jd )
     double jd; 
{

    double 	time;		/* unix-time, seconds since 1970.0 */
    Coord3d 	earth;		/* position of earth */ 
    Coord3d	position; 	/* position of a body */ 
    Coord3d	velocity;	/* its velocity */ 
    PCoord	rpos; 
    int 	index;		/* body index to jpl ephemeris */ 

    /* ---------------------------------------------------------------- */ 
    printf(" ecliptic positions from Explanatory Supplement orbits  \n" );  
    printf("  coords of J2000.0 \n" ); 
 
    time = calendarJulianToUnix( jd );
 
    printf(" JD %12.2f %s \n", jd, fvfJulianToDate( jd ));	
 
    ExplanatorySupplementOrbits( jd ); 

    for ( index=0; index<9; ++index )
    {
	navKepler( planet[index], time, &position ); 
	GeoScalarMultVector( 1./AU, &position, &position ); 
	
	matRectToPolar( &position, &rpos ); 
	rpos.pc_long = normalize( rpos.pc_long, 0., PI2 ); 
	tabulateDegrees( planetName[index], &rpos ); 
    }
 
	printf(" \n");
 

    /* ---------------------------------------------------------------- */
    printf(" equatorial positions from Explanatory Supplement orbits  \n" );  
    printf(" uncorrected for aberration or light-delay \n" ); 
    printf(" no corrections for precession or nutation \n" ); 

    navKepler( planet[EARTH], time, &earth ); 
    GeoScalarMultVector( 1./AU, &earth, &earth ); 
    cliptoquat( &earth ); 

    time = calendarJulianToUnix( jd );
    
    printf(" JD %12.2f %s \n", jd, fvfJulianToDate( jd ));	
 
    ExplanatorySupplementOrbits( jd ); 

    for ( index=0; index<9; ++index )
    {
	quatPos( planet[index], time, &position, &velocity ); 
	GeoSubtractVector( &position, &earth, &position ); 
	  
	if (index == EARTH)
	{
	    continue; 
	}

	matRectToPolar( &position, &rpos ); 
	rpos.pc_long = normalize( rpos.pc_long, 0., PI2 ); 
	tabulateHours( planetName[index], &rpos ); 

    }
 
    moonRadec( jd, &rpos ); 
    rpos.pc_long = normalize( rpos.pc_long, 0., PI2 ); 
    rpos.pc_r /= AU; 
    tabulateHours( "moon   ", &rpos ); 
    
    sunRadec( jd, &rpos ); 
    rpos.pc_long = normalize( rpos.pc_long, 0., PI2 ); 
    rpos.pc_r /= AU; 
    tabulateHours( "sun    ", &rpos ); 

    printf(" \n");
 
}
 
 


/* ---------------------------------------------------------------- */ 
/* tabulateHours() and tabulateDegrees()
 * 	print a line in the tables put out by fastPlanets() above
 * inputs
 *	a name or label (character string)
 *	position in polar coordinates
 * outputs
 *	none
 * side effects
 * 	prints
 * notes
 *	these routines are a little different from the corresponding
 *	routines in planets.c. they probably should be the same.
 * ----------------------------------------------------------------
 */ 

void 
tabulateHours( name, pol )
     char *name;
     PCoord *pol;
{
  double astro_long; 

  printf("       %s    ", name ); 
  astro_long = normalize( pol->pc_long, 0., 2.*PI);
  printHoursMS(astro_long);
  printf(" | "); 
  printDegsMS(pol->pc_lat);
  printf(" r = %f \n", pol->pc_r );

}

/* ---------------------------------------------------------------- */ 
void 
tabulateDegrees( name, pol )
     char *name;
     PCoord *pol;
{
  double astro_long; 

  printf("       %s    ", name ); 
  astro_long = normalize( pol->pc_long, 0., 2.*PI);
  printDegsMS(astro_long);
  printf(" | "); 
  printDegsMS(pol->pc_lat);
  printf(" r = %f \n", pol->pc_r );

}

