/* 
 * planets.c prints positions for planets, sun and moon 
 * from orbits or from ephemeris data files 
 * andrew porter, 1994 
 */
 

#include <stdio.h>
#include <stdlib.h> 
#include <math.h>
#include <time.h>
 
 
#include "../include/ms.h"
#include "../include/matrix.h"
#include "../include/hnav.h"
 
 
/* these are in exorbs.c: */ 
extern Orbit *planet[];
extern char  *planetName[];
 
 
/* routines in this file: */  
void eclipticPlanets();
void rawData(); 
void equatorialPlanets(); 
void fixRequest(); 
void usage();  
void tabulateHours(); 
void tabulateDecDegrees(); 
int  indexATOI(); 
char *bodyname(); 

/* coord systems: */ 
#define ECLIPTIC 1 
#define EQUATORIAL 2 
 
/* coord styles / which subroutine does the work: */ 
#define RAW 1 
#define EARTH_RADEC 3 	/* long in hours, m,s; lat in degs, min, secs */ 
#define BARY_ECLIPTIC 5 

/* data source 
 * ORBITS and TABLES are #define'd in almanac.h; 
 * #define ORBITS 0	orbits from Expl. Suppl. to the Almanac 
 * #define TABLES 1 	tables, as spec. in CATALOGFILE 
 */ 
 
#define BODY_COUNT 15 
 
typedef struct request {
    int coordstyle; 		/* which style of printout */
    int coordsys;		/* for raw data, eclip./equat coords */

    double startdate;		/* start date */ 
    double stopdate;		/* stop  date */ 
    double date_increment;	/* increment between printout times */
    int data_source;		/* JPL ephemeris tables or keplerian orbits*/
    char body[20];		/* char string for list of bodies */ 
} Request; 
 
 
/* ----------------------------------------------------------------
 * main reads command-line requests and passes them on 
 * to data-reporting routines 
 * ----------------------------------------------------------------
 */
 
main( argc, argv )
     int argc;
     char *argv[];
{
    Request wish; 	/* collection of requests for data */ 
 
    fixRequest( argc, argv, &wish ); 
    
    if( wish.coordstyle == BARY_ECLIPTIC )  eclipticPlanets( &wish ); 
    if( wish.coordstyle == EARTH_RADEC )    equatorialPlanets( &wish ); 
    if( wish.coordstyle == RAW )            rawData( &wish ); 
 
    return 0; 
 
}
 
 

/* ----------------------------------------------------------------
 * fixRequest parses the command line
 *	and interprets the requests in it,
 *	putting them in the structure pointed to by its 3rd arg
 * inputs
 * 	count and pointer to the command line args
 *	pointer to a Request data-structure
 * outputs
 *	none
 * side effects
 * 	fills in the Request
 * 	according to requests on the command line 
 * ----------------------------------------------------------------
 */


void 
fixRequest( argc, argv, wish )
  int argc;
  char *argv[];
  Request *wish;
{
    double jd;  	/* julian day number of current GMT */ 
    int i;	/* loop control */ 
    int count; 	/* to count your way along the argument list */ 
    int year;
    int month;
    int iday;
    double fday;

 
    /* -------------------------------- */ 
    jd = xjdnow(); 


    /* -------------------------------- */ 
    /* default requests: */ 
    wish->coordstyle = EARTH_RADEC; 
    wish->coordsys = ECLIPTIC; 

    wish->startdate = jd;
    wish->stopdate = jd;
    wish->date_increment = 1.; 
    wish->data_source = TABLES; 
 
    /* default bodies requested: planets, sun, moon: */ 
    for (i=0; i<BODY_COUNT; ++i)
    {
	wish->body[i] = 0;
    }
    strcpy( wish->body, "012345678ms" ); 
  
    /* ---------------------------------------------------------------- */
    /* process command line args, one group at a time: */ 

    count = 1; 
    while( count<argc )
    {	
	if (strcmp( "jd", argv[count] ) == 0
	    ||
	    strcmp( "startjd", argv[count] ) == 0)
	{
	    if (argc <= count+1) usage(); 
	    jd = atof( argv[count+1] );
	    wish->startdate = jd; 
	    wish->stopdate  = wish->startdate; 
	    count += 2;
	    continue; 
	}
	if (strcmp( "stopjd", argv[count]) == 0) 
	{
	    if (argc <= count+1) usage(); 
	    jd = atof( argv[count+1] );
	    wish->stopdate = jd; 
	    count += 2;
	    continue;
	}
	if (strcmp( "date", argv[count] ) == 0
	    ||
	    strcmp( "startdate", argv[count] ) == 0) 
	{
	    if (argc <= count+3) usage(); 
	    year = atoi( argv[count+1] );
	    month = atoi( argv[count+2] );
	    fday = atof( argv[count+3] );
	    iday = fday; 
	    jd = calendarDateToJulian( year, month, iday) + fday - iday;
	    wish->startdate = jd; 
	    wish->stopdate  = wish->startdate; 
	    count += 4;
	    continue; 
	}
	if (strcmp( "stopdate", argv[count] ) == 0)
	{
	    if (argc <= count+3) usage(); 
	    year = atoi( argv[count+1] );
	    month = atoi( argv[count+2] );
	    fday = atof( argv[count+3] );
	    iday = fday; 
	    jd = calendarDateToJulian( year, month, iday) + fday - iday;
	    wish->stopdate = jd; 
	    count += 4;
	    continue; 
	}
	if (strcmp( "orbits", argv[count] ) == 0)
	{
	    wish->data_source = ORBITS;
	    ++count; 
	    continue;
	}
	if (strcmp( "tables", argv[count] ) == 0)
	{
	    wish->data_source = TABLES;
	    ++count;
	    continue; 
	}
	if (strcmp( "days", argv[count] ) == 0)
	{
	    wish->date_increment = atof( argv[count+1] );
	    count += 2;
	    continue;
	}
	if (strcmp( "seconds", argv[count] ) == 0)
	{
	    wish->date_increment = atof( argv[count+1] )/86400.;
	    count += 2;
	    continue;
	}
	if (strcmp( "bodies", argv[count] ) == 0)
	{
	    strcpy( wish->body, argv[count+1] );
	    count += 2;
	    /* printf(" body = %s \n", wish->body ); */ 
	    continue;
	}

	/* coord system, applicable only 
	   for "raw" and "bary-centered" options: */ 
	if (strcmp( "ecliptic", argv[count] ) == 0 )
	{
	    wish->coordsys = ECLIPTIC;
	    ++count; 
	    continue;
	}
	if (strcmp( "equatorial", argv[count] ) == 0 )
	{
	    wish->coordsys = EQUATORIAL;
	    ++count; 
	    continue;
	}

	/* options for coord-style, coord-system, and center: */ 
	if (strcmp( "raw", argv[count] ) == 0)
	  {
	      wish->coordstyle = RAW;
	      ++count; 
	      continue;
	  }
	if (strcmp( "bary-centered", argv[count] ) == 0 )
	{
	    wish->coordstyle = BARY_ECLIPTIC;
	    ++count; 
	    continue;
	}
	if (strcmp( "apparent", argv[count] ) == 0 
	    ||
	    strcmp( "geo-centered", argv[count] ) == 0 
	    )
	{
	    wish->coordstyle = EARTH_RADEC;
	    ++count; 
	    continue;
	}


	/* this arg makes no sense: */ 
	printf(" arg %s makes no sense \n", argv[count] );
	usage(); 
	++count;

    }
 
}
 
 
 
 


void usage()
{
    printf(" arguments do not parse successfully; \n" );
    printf(" planets [no arguments] gets you planet positions for current time \n"); 
    printf(" planets jd julian-day-number gets positions for that julian date \n");
    printf(" planets startjd julian-day-number gets positions for that julian date \n");
    printf(" planets date year month day gets positions for that date \n");
    printf(" planets startdate year month day gets positions for that date \n");
    printf(" planets stopdate year month day does positions until stopdate \n");
    printf(" planets bodies [0-9ms] does only the bodies indicated \n"); 
    printf(" planets orbits gets positions from orbits, not tables \n");
    printf(" planets tables gets positions from JPL ephemeris tables \n");
    printf(" planets days n prints at interval of n days \n");
    printf(" planets seconds n prints at intervals of n seconds \n");
    printf(" planets raw prints raw cartesian data, position and velocity \n" ); 
    printf(" planets raw equatorial gets same, but in equatorial coords \n" );
    printf(" (default for raw option is ecliptic coords) \n" ); 
    printf(" you can combine as many groups of argument-commands as you like \n"); 

    exit(0);
}



 

/* ----------------------------------------------------------------
 * this prints out ecliptic planet positions per the request in its argument
 * 	this routine should be omitted, 
 * 	and its function subsumed under equatorialPlanets as a special case,
 * 	but it is a useful model of a simple routine ... 
 * inputs
 *	a request for position data
 * outputs
 *	none
 * side effects
 * 	prints barycentered planetary positions 
 *	per request in its argument;
 *	for which, see comments and code at fixRequest(), above 
 * ----------------------------------------------------------------
 */ 



void 
eclipticPlanets( command )
     Request *command; 
{
    double jd;
    Coord3d	position; 	/* position of a body */ 
    Coord3d	velocity;	/* its velocity */ 
    int 	success;	/* return flag from subroutine calls */ 
    int 	index;		/* body index to jpl ephemeris */ 
    int 	center; 	/* center-index */ 
    char	*s; 		/* ptr to command->body[] list */ 
    static char	*catfilename; 
    char        *name;          /* name of body */ 
 
    if (command->data_source == TABLES)
    {
	catfilename = getEphemerisCatalog(); 
	printf(" catalog file name = %s \n", catfilename );  
    }

    printf(" solar-system barycentric ecliptic positions: \n");

    for( jd = command->startdate; 
	jd <= command->stopdate; 
	jd+=command->date_increment )
    {

 
	if (command->data_source == ORBITS)
	  ExplanatorySupplementOrbits( jd );        /* sets orbits in exorbs.c */ 

	printf(" JD %12.2f %s ", jd, fvfJulianToDate( jd ));	
	if( strlen( command->body ) > 1 )
	  printf("\n"); 

	for( s = command->body; *s; ++s )  
	{
	    index = indexATOI( *s ); 
	    name  = bodyname( index ); 

	    if (command->data_source == TABLES)
	      success = jplBody( jd, catfilename, 
			      index, BARYCENTERED, AUDAY, 
			      &position, &velocity ); 
	    else if (command->data_source == ORBITS)
	      quatPos( planet[index], calendarJulianToUnix( jd ), 
		      &position, &velocity ); 

	    if ( command->coordsys == ECLIPTIC)
	      toEcliptic( 2451545.0, &position ); 

	    tabulateDecDegrees( name, &position );
	    if ( strlen( command->body ) > 1 )
	      printf("\n"); 
	}
 
	printf(" \n");
 
    } 			/* for days */ 
 
}
 
 

/* ----------------------------------------------------------------
 * this prints out raw positions and velocities 
 * 	in cartesian coordinates 
 * inputs
 *	a request for position data
 * outputs
 *	none
 * side effects
 * 	prints barycentered planetary positions 
 *	in equatorial coordinates of J2000.0;
 *	per request in its argument, 
 *	for which, see comments and code at fixRequest(), above 
 * ----------------------------------------------------------------
 */ 



void 
rawData( command )
     Request *command; 
{
    double jd;
    Coord3d	position; 	/* position of a body */ 
    Coord3d	velocity;	/* its velocity */ 
    int 	success;	/* return flag from subroutine calls */ 
    int 	index;		/* body index to jpl ephemeris */ 
    int 	center; 	/* center-index */ 
    char	*s; 		/* ptr to command->body[] list */ 
    static char	*catfilename; 
    char        *name;          /* name of body */ 
 
    if (command->data_source == TABLES)
    {
	catfilename = getEphemerisCatalog(); 
	printf(" catalog file name = %s \n", catfilename );  
    }

    printf(" solar-system barycentric ecliptic positions: \n");

    for( jd = command->startdate; 
	jd <= command->stopdate; 
	jd+=command->date_increment )
    {
	if (command->data_source == ORBITS)
	  ExplanatorySupplementOrbits( jd );        /* sets orbits in exorbs.c */ 

	printf(" JD %12.2f %s ", jd, fvfJulianToDate( jd ));	
	if( strlen( command->body ) > 1 )
	  printf("\n"); 

	for( s = command->body; *s; ++s )  
	{
	    index = indexATOI( *s ); 
	    name = bodyname( index ); 

	    if (command->data_source == TABLES)
	      success = jplBody( jd, catfilename, 
			      index, BARYCENTERED, AUDAY, 
			      &position, &velocity ); 

	    else if (command->data_source == ORBITS)
	      quatPos( planet[index], calendarJulianToUnix( jd ), 
		      &position, &velocity ); 

	    if ( command->coordsys == ECLIPTIC )
	    {
		toEcliptic( 2451545.0, &position ); 
		toEcliptic( 2451545.0, &velocity ); 
	    }

	    printf(" %s ", name ); 
	    printf(" pos: % 20.15f % 20.15f %20.15f ", 
		   position.x, position.y, position.z );
	    printf(" vel: % 20.15f % 20.15f %20.15f ", 
		   velocity.x, velocity.y, velocity.z );
	    if ( strlen( command->body ) > 1 )
	      printf("\n"); 
	}
 
	printf(" \n");
 
    } 			/* for days */ 
 
}
 
 

/* ----------------------------------------------------------------
 * to print apparent place of planets: 
 * 	it will print apparent place, which can be checked 
 * 	against the printed tables in the Almanac 
 *	if you use data from the de-200 ephemeris, 
 *	agreement should be very close.  
 *
 * inputs
 *	request for positions; see comments at fixRequest(), above
 * outputs
 *	none
 * side effects 
 * 	prints positions 
 *
 * data sources, apparent place calculations: 
 *	see planets.doc 
 * limitations, approximations
 *      the moon apparent place calculation 
 *      does not account for time delay;
 *      the sun's time-delay is approximate 
 * ----------------------------------------------------------------
 */
 
 
void 
equatorialPlanets( command ) 
     Request *command; 
{
    double jd;
    WhichPositionData data;	/* data package for apparentPlanetPosition() */ 
    double 	time;		/* unix-time, seconds since 1970.0 */
    Coord3d 	geqpos;		/* geo-centric equatorial position of planet */
    Coord3d	velocity;	/* this is a dummy for calls to jplBody */ 
    PCoord  	pol;		/* polar coord of planet */
    int 	success;	/* return flag from subroutine calls */ 
    char	*s;		/* ptr to command->bodi[], body list */ 
    int 	index;		/* body index */ 

    matrix3d precessNutate; 
    static char	*catfilename; 
 
    if (command->data_source == TABLES)
      catfilename = getEphemerisCatalog();
 
    printf(" apparent place, RA-dec: \n"); 

    for( jd = command->startdate; 
	jd <= command->stopdate; 
	jd += command->date_increment )
    {
	time = calendarJulianToUnix( jd );
 
	if (command->data_source == ORBITS)
	  ExplanatorySupplementOrbits( jd );        /* sets orbits in exorbs.c */ 
 
	printf(" JD %10.1f %s ", jd, fvfJulianToDate( jd ));	
	if ( strlen( command->body ) > 1 )
	  printf("\n"); 

	for( s = command->body; *s; ++s )  
	{
	  index = indexATOI( *s ); 
 
	    if (index == EARTH || index == EARTHINDEX) 
	      continue;	/* skip the earth */
 
	    /* 0-8 = merc-pluto, i. e., jpl's 0-indexing */ 
	    /* jpl ephemeris positions */ 
	    if ( index >= MERCURY && index <= PLUTO )
            {
		data.flag = command->data_source; 
		if (command->data_source == TABLES)
		  data.catfile = catfilename;
		data.body = index; /* jpl numbers planets 0-8 */ 

		if (command->data_source == ORBITS) 
		  data.orb = planet[index];   
		  /* (these pointers and orbits are in exorbs.c) */ 

		apparentPlanetPosition( &data, time, &geqpos ); 	

		tabulateHours( planetName[index], &geqpos ); 

	    }
 
	    if ( index == MOON )	/* jpl sun and moon */
            {
		/* ------------ the moon -------------- */ 
		if (command->data_source == TABLES)
		{
		    success = jplBody( jd, catfilename, 
				      MOONINDEX, EARTHCENTERED, AUDAY, 
				      &geqpos, &velocity ); 
		    /* not corrected for light delay, about 1.25 seconds */ 
		    /* corr for precession and nutation */ 
		    almanacEquinox( time, precessNutate ); 
		    matMatVV( precessNutate, &geqpos, &geqpos ); 
		}
		if (command->data_source == ORBITS)
		{   /* coords of date; Montenbruck is none too clear */ 
		    moonRadec( jd, &pol ); 
		    matPolarToRect( &pol, &geqpos );
		}

		tabulateHours( "moon ", &geqpos ); 
 
	    }
	    if ( index == SUN )
	    {

		/* ------------ the sun --------------- */ 
		/* note here we DO correct for time delay: */ 
		if (command->data_source == TABLES)
		{
		    success = jplBody( jd - 8.316/1440., catfilename, 
				      SUNINDEX, EARTHCENTERED, AUDAY, 
				      &geqpos, &velocity ); 
		    almanacEquinox( time, precessNutate ); 
		    matMatVV( precessNutate, &geqpos, &geqpos ); 
		}
		if (command->data_source == ORBITS)
		{   /* coords of date? Montenbruck is silent */ 
		    sunRadec( jd, &pol ); 
		    matPolarToRect( &pol, &geqpos );
		}

		tabulateHours( "sun  ", &geqpos ); 
 
	    }
	} 		/* for planets */ 
      if ( strlen( command->body ) > 1 ) 
	printf(" \n");
    } 			/* for days */ 
 
}
 

/* ---------------------------------------------------------------- */ 
/* tabulateHours() and tabulateDecDegrees()
 *	print positions in hours (RA-dec) and degrees (ecliptic positions)
 * inputs
 *	a character string (planetary name, usually)
 *	a position in cartesian coordinates
 * outputs
 *	none
 * side effects
 * 	prints name and position in polar coordinates 
 */ 



void 
tabulateHours( name, position )
     char *name;
     Coord3d *position; 
{
    PCoord polar;
    double astro_long; 

    matRectToPolar( position, &polar );

    printf(" %12s ", name ); 
    astro_long = normalize( polar.pc_long, 0., 2.*PI);
    printHoursMS(astro_long);
    printf(" | "); 
    printDegsMS(polar.pc_lat);
    printf(" r = %f \n", polar.pc_r );

}



void 
tabulateDecDegrees( name, pos )
     char *name; 
     Coord3d *pos;
{
    PCoord rpos; 

    matRectToPolar( pos, &rpos ); 
    rpos.pc_long = normalize( rpos.pc_long, 0., PI2 ); 
    printf(" % 10s  long % 8.3f lat % 8.3f, r % 6.1f AU ", 
	   name, 
	   rpos.pc_long * TODEGREES, 
	   rpos.pc_lat * TODEGREES,
	   rpos.pc_r ); 
    

}



/* ----------------------------------------------------------------
 * indexATOI translates from the ascii command-line index code (0-8,ms)
 *       to the integer index convention in jpl.h
 * inputs
 *       a one-character ascii body code
 * outputs
 *       an integer body code
 * side effects
 *       none
 * ----------------------------------------------------------------
 */ 


int
indexATOI( c )
     char c;
{

  /* in this file, we want earth, not Earth-Moon Barycenter: */ 
  if ( c == '2' )
    return EARTHINDEX;     

  if ( c >= '0' && c <= '8')
    return (c - '0');
  else if ( c == 'm')
    return MOON;
  else if ( c == 's')
    return SUN;
  else 
    {
      printf(" you called indexATOI with arg %c \n", c );
      printf(" this is not a valid argument \n");
    } 

  return 0; /* default has to be something -- mercury */ 

}


char * 
bodyname( index )
     int index; 
{

  if ( index == EARTHINDEX )
    return planetName[2]; 

  if ( index <= 8 )
    return planetName[index]; 
  else if ( index == MOON )
    return "moon   "; 
  else if ( index == SUN )
    return "sun    ";
  else 
    return "no-body";

}

