/* 
 * apparent.c contains routines for the planetary positions 
 *	apparentPlanetPosition() manages planetary apparent place calculation
 *	precession and nutation are handled in routines in precnuta.c
 * 91/11/25 ff. 
 *	copyright (c) 1991, 1994 Andrew P. Porter 
 */

#include <stdio.h>
#include <math.h>

#include "../include/ms.h"
#include "../include/matrix.h"
#include "../include/hnav.h" 

extern char *malloc();
extern double normalize();

void greenAberration();
void quatPos();
void earthmoonToEarth();

double xtrueAnomaly();  

/* this is "public" storage */
extern Orbit almanacOrbit_Mercury;
extern Orbit almanacOrbit_Venus;
extern Orbit almanacOrbit_Earth;
extern Orbit almanacOrbit_Mars;
extern Orbit almanacOrbit_Jupiter;
extern Orbit almanacOrbit_Saturn;
extern Orbit almanacOrbit_Uranus;
extern Orbit almanacOrbit_Neptune;
extern Orbit almanacOrbit_Pluto;
extern Orbit almanacOrbit_PlanetX;

extern Orbit *planet[10];
extern char  *planetName[10];



/* ----------------------------------------------------------------
 * apparentPlanetPosition
 * 	geocentric coordinates, cf. Almanac, pp. E-14 ff.
 *	gets position in ecliptic coords from orbital elements referred to ecliptic 
 *	transforms to heliocentric equatorial coords
 *	transforms to geocentric   equatorial coords
 * inputs
 *	orbit, 
 *	time, 
 *	place to put position
 * outputs
 *	none
 * side effects
 *	puts position where you tell it to 
 * method
 *	find planet position
 *	find planet position, retarded for light transit time
 *	correct for aberration (a relativistic effect)
 *	correct for precession and nutation
 * formulas for <real place to apparent place> taken 
 *	from the _Astronomical Almanac_, pp. B36-37
 * limitations -- 
 *      does not know planetary nutation, 
 *      nor earth-rotation axis wobble,
 *      or other high-order effects in USNO NOVAS code.  
 *      used with the DE-200 ephemeris, it's as good as 
 *          the tables in the printed almanac, differing
 *          by at most 1 in the last place.  
 * ----------------------------------------------------------------
 */

void
apparentPlanetPosition( data, utime, equatPos)
     WhichPositionData  *data;	/* contains orbit pointer or catalog filename and flag to tell which */ 
     double utime;		/* unix-time for which position is requested */
     Coord3d *equatPos;		/* geocentric  equatorial planetary position */
{
    double 	time;		/* time in seconds since 1988.0 */
    double	jd;		/* julian date for time utime */
    Coord3d 	earthPos;	/* heliocentric equatorial earth position */
    Coord3d 	hecPos;		/* heliocentric equatorial planetary position */
    matrix3d	ecliptic;	/* transform matrix from ecliptic to equatorial coords */
    matrix3d	equinox;	/* transfrom from coords of 2000.0 to coords of date */
    Coord3d	gecPos2000;	/* position in equatorial coords of J 2000.0 */  
    Coord3d	geqPos;		/* same,  coords uncorr'd for precession & nutation */
    Coord3d 	pvel;		/* planet velocity */
    Coord3d 	evel;		/* earth  velocity */
    double  	V;		/* magnitude of evel */
    Coord3d 	dvel;		/* diff of velocities */
    Coord3d	aberration;	/* the aberration correction */
    Coord3d	retard;		/* the retardation correction to planet position */
    double	ptrue;		/* planet true anomaly */
    double 	etrue;		/* earth true anomaly  */
    double	lightTime;	/* light transit time from planet to earth */
    double	lightspeed;	/* speed of light in AU per day */
    Coord3d	sumvec;		/* temp storage */ 
    Coord3d	earth;		/* earth position, equat. coords of J 2000.0 */
    Coord3d	equatp;		/* a dummy equatorial position */
    Coord3d	e, p, q;	/* unit vectors of p. B37 */
    Coord3d	p1, p2, p3;	/* more unit vectors of p. B37 */
    double      E, P, Q;	/* scalars of p. B36 */
    double 	muconst;	/* 2.*mu/c*c; see pp. B36-37 */
    double 	beta;		/* what is usually called gamma . . . */
    double  	inv_beta;	/* scalar 1./beta */
    double 	temp;
    double  	temp2;
    double 	distance;	/* distance in AU from earth to planet */ 
    Coord3d	tempVec;	/* temporary storage */
    int 	success;	/* whether called routines worked or not */ 

    time = utime;
    lightspeed = 173.1446333;	/* from usno aberrat. subr, in novas.f; units AU/day */
    muconst = 2. * 9.87e-9;	/* units are AU; value is on p. B36 */
    jd = calendarUnixToJulian( time ); 

/* ---------------------------------------------------------------- */
/* get retarded positions */ 

    if (data->flag == ORBITS)
    {
	quatPos( data->orb, time, &hecPos, &pvel ); 
	quatPos( &almanacOrbit_Earth, time, &earthPos, &evel );
	earthmoonToEarth( time, &earthPos );
    }
    if (data->flag == TABLES)
    {
	success = jplBody( jd, data->catfile, data->body, 
 		SUNCENTERED, AUDAY, &hecPos, &pvel );
	if (success == 0)
	{
	    printf(" jplBody bombed, so apparentPlanetPosition crashes \n");
	    exit(1); 
	}
	/* jplBody gives earth directly, not earth-moon barycenter */ 
	success = jplBody( jd, data->catfile, EARTHINDEX, 
		SUNCENTERED, AUDAY, &earthPos, &evel );
	if (success == 0)
	{
	    printf(" jplBody bombed, so apparentPlanetPosition crashes \n");
	    exit(1); 
	}
	/* jplBody gives earth directly, not earth-moon barycenter */ 
	/* so we don't have to call earthmoonToEarth() */ 

    }

    /* geocentric equatorial planet position: */
    GeoSubtractVector( &hecPos, &earthPos, &gecPos2000);   

    distance = GeoVectorLength( &gecPos2000 );
    lightTime = distance / lightspeed; 

    E = GeoVectorLength( &earthPos );
    P = GeoVectorLength( &gecPos2000 );
    Q = GeoVectorLength( &hecPos );

    lightTime += muconst * log( (E+P+Q) / (E-P+Q) ) / lightspeed;

    /* N. B., lightTime here is in days; argument to navTrueAnomaly is in seconds */
    /* note that quatPos is called with time in seconds; jplBody with time in days */
    if (data->flag == ORBITS)
    {
	quatPos( data->orb, time - lightTime*86400., &hecPos, &pvel ); 
    }
    if (data->flag == TABLES)
    {
	jplBody( jd - lightTime, data->catfile, data->body, 
		SUNCENTERED, AUDAY, &hecPos, &pvel );
    }

/* ---------------------------------------------------------------- */
/* the remaining correctionns */ 

    /* geocentric ecliptic planet position */
    GeoSubtractVector( &hecPos, &earthPos, &gecPos2000);   

    /* aberration in two steps, p. B37: */
    E = GeoVectorLength( &earthPos );
    P = GeoVectorLength( &gecPos2000 );
    Q = GeoVectorLength( &hecPos );
    
    GeoUnitVector( &earthPos, 	&e );
    GeoUnitVector( &hecPos, 	&q );
    GeoUnitVector( &gecPos2000, &p );

    GeoScalarMultVector( - GeoDotProduct( &e, &p), &q, &sumvec );
    matAccumulateVector(   GeoDotProduct( &p, &q), &e, &sumvec );
    GeoScalarMultVector( muconst/(E*(1. + GeoDotProduct( &q, &e ))), &sumvec, &sumvec );
    GeoAddVector( &p, &sumvec, &p1 );


    /* what we think we are doing here: evaluating the formula for p2, p. B37
     *	the parenthese there are hideous; here is what I think they mean:
     *			       -1                    (p1*V)	
     *			   beta   * p1   +  [ 1  +   -------   ] * V
     *			                   	           -1
     *			                            (1+beta  )  
     *
     *	p2     =     ------------------------------------------------------------
     *					1  + p1*V
     *
     */

    GeoScalarMultVector( 1./lightspeed, &evel, &evel );
    V = GeoVectorLength( &evel );

    beta = 1. / (sqrt( 1. - V*V ) );
    inv_beta = 1. / beta; 

    temp = GeoDotProduct( &p1, &evel );
    temp2 = 1.+temp/(1.+inv_beta);


    GeoScalarMultVector( temp2, &evel, &sumvec );
    matAccumulateVector( inv_beta, &p1, &sumvec );
    GeoScalarMultVector( 1./(1.+temp), &sumvec, &p2 );
    GeoUnitVector( &p2, &p2 );
    GeoScalarMultVector( distance, &p2, &p2 );


    /* ---------------------------------------------------------------- */
    /* get the precession and nutation matrices: */

    almanacEquinox(  utime, equinox );
    matMatVV( equinox, &p2, equatPos );


}

