

#include <stdio.h>
#include <math.h>
#include <time.h>

#include "../include/ms.h"
#include "../include/matrix.h"
#include "../include/hnav.h"

/* NOTE if you port to DOS, esp. with Borland C, 
 * many int's should become long,
 * and %d on output should become %ld
 * fvfJulianToDate in particular ... 
 * or else date conversions will be very very wrong
 */ 



/* ----------------------------------------------------------------
 * calendarDateToJulian returns Julian date for year, month, date
 * inputs
 *	year, month, date of month
 * outputs
 *	double-float Julian date (1961/01/01 0h UT is JD 2437300.5)
 * side effects
 *	none
 * sources
 *	for now, p. 69 of Martin V. Zombeck (of SAO), 
 *		Handbook of Space Astronomy and Astrophysics,
 *		Cambridge U. P., 1982
 * 	he cites
 *		H. F. Fliegel and T. C. van Flandern, Comm ACM 11 (1968) 657
 *	_Explanatory Supplement_ (1992), p. 604 has a very similar formula
 * 	and credits the same article by Fliegel and van Flandern 
 * cautions
 *	I had to modify this routine (see notes below) somewhat to make it come out right
 * ----------------------------------------------------------------
 */

double 
calendarDateToJulian( syear, smonth, sday)
     int syear;
     int smonth;	/* january = 01 */
     int sday;		/* january 01 = month 1, day 1, not day 0 */
{
    long jd;	/* the julian date to be calculated */
    long year;
    long month;
    long day;

    year = syear;
    month = smonth;
    day = sday; 

    jd  = - 32075 + 1461 * (year + 4800 + (month - 14) / 12) / 4; 		
    jd += 367 * (month - 2 - (month - 14)/12 * 12 ) / 12;
    jd -= 3 * ((year + 4900 + (month - 14) /12) /100 ) /4;
    jd += day;
      
    /* jd of a date begins at noon previous day; 
     * see Astromonical Almanac 1988, pp. B8 ff.: 
     * I had trouble getting this routine to work right; 
     * this fudge makes it come out correctly 
     */
    
    return ((double) jd) - 0.5; 

}



/* ----------------------------------------------------------------
 * fvfJulianToDate is Fliegel and Van Flandern's algorithm
 * 	it is good outside the range (1902,2038); use it freely 
 * inputs
 *	julian date, as a double-float 
 * outputs
 *	ptr to gregorian calendar date string
 * side effects
 *	sets calendar date string pointed to on input
 * method
 *	algorithm of H. F. Fliegel and T. C. van Flandern 
 * cautions
 *	use the answer before you loose it . . . it is in static storage here
 * Source 
 *	Henry F. Fliegel, Thomas C. Van Flandern, "Machine Algorithm 
 *	for Processing Calendar Dates," _Commmunications of the ACM_, 11 (1968) 657 
 *	or see _Explanatory Supplement_ (1992), p. 604.  
 * ----------------------------------------------------------------
 */

char * 
fvfJulianToDate( jd )
     double  jd;	/* double float Julian date */
{
    double jdnoon;	/* jd at noon on the day in question */ 
    int L,N;
    int year;
    int month; 	/* month = 1-12 */ 
    int day; 	/* a presume 1 to 28-31 */ 
    static char nice[100];	/* to hold the answer */
    double fraction; 	/* fraction of day */ 
    int hour, minute, second;		/* time of day in integers */ 
    double fhour, fminute, fsecond;	/* time of day in double-floats */ 

    jdnoon = (double) (int) ( jd + 0.5 ); 
    fraction = jd - (jdnoon - 0.5 );

    /* Fliegel and Van Flandern's algorithm: */ 
    L = jdnoon + 68569;
    N = 4 * L / 146097; 
    L = L - (146097*N + 3) / 4; 

    year = 4000 * (L+1) / 1461001; 
    L = L - 1461 * year / 4  +  31; 
    month = 80 * L / 2447; 
    day = L - 2447 * month / 80;
    L = month / 11;
    month = month + 2 - 12 * L;
    year = 100 * ( N - 49)  +  year + L;
    /* end of F & vF's algorithm */ 

    fhour = fraction * 24.;
    hour = (int) fhour;
    fminute = (fhour - hour) * 60.;
    minute = (int) fminute; 
    fsecond = (fminute - minute) * 60.; 
    second = (int) fsecond; 

    /* now format the time in struct t to string nice: */
    (void) sprintf( nice, "  %d/%2d/%2d %2d:%2d:%2d", year, 
		  month, day, hour, minute, second ); 

    return nice;	/* for now . . . better later */

}


/* ----------------------------------------------------------------
 * calendarUnixToJulian() converts unix-time to julian date
 * inputs
 *	unix time, double-float, seconds since 1970.0
 * outputs
 *	julian day, double-float
 * side effects
 * 	none
 * source: JD for 1970.0 = 2440587.5, cf. p. 439 of 
 *	Explanatory Supplement to the Astronomical Ephemeris, 1961
 * 
 * calendarJulianToUnix() converts julian day to unix-time
 * inputs
 *	julian day, DOUBLE	
 * outputs
 *	unix-time, seconds since 1970.0
 * side effects
 *	none
 * ----------------------------------------------------------------
 */


double
calendarUnixToJulian(utime)
     double utime;
{
  double jd; 

  jd = (utime / 86400.) + 2440587.5;

  return jd; 
}


double 
calendarJulianToUnix( jd )
     double jd;		/* julian day */
{
    double x;

    x = (jd - 2440587.5) ;
    x *= 86400.;

    return x; 
}


/* ----------------------------------------------------------------
 * GMT() -
 *
 * Return double GMT seconds since 1970 ala Unix.  This is civil
 * time in the usual Unix format of seconds since 00:00:00, Jan 1, 1970.
 * Fractional seconds are allowed.
 *
 * NOTE: month and day are in the ranges [1,12] and [1,31] respectively.
 *       year should be either [0,99] or [1970,2020(or so)].
 * not sure whether this is used here or not.
 * author -- Gary Preckshott 
 */

#define BASEYEAR 1970
#define BASELEAP 1968
#define YEAR 365
#define DAY 24
#define HOUR 60
#define MINUTE 60

static int monthdays[] = {31,28,31,30,31,30,31,31,30,31,30,31};
static int leapmonthdays[] = {31,29,31,30,31,30,31,31,30,31,30,31};

double
GMT( year, month, day, hour, minute, second )
    int		year, month, day, hour, minute;
    double	second;
{
    double 	seconds = 0.0;
    int		yr,mo;

    if (year < 1900) year += 1900;
    if (year < BASEYEAR) return -1.0;
    for (yr=BASEYEAR;yr<year;yr++)
    {
	seconds += YEAR * DAY * HOUR * MINUTE;
	if ( ((yr - BASELEAP) % 4) == 0 )
	{
	    seconds += DAY * HOUR * MINUTE;
	}
    }
    if ( ((year - BASELEAP) % 4) == 0 )
    {
	for (mo=1;mo<month;mo++)
	    seconds += leapmonthdays[mo-1] * DAY * HOUR * MINUTE;
    }
    else
    {
	for (mo=1;mo<month;mo++)
	    seconds += monthdays[mo-1] * DAY * HOUR * MINUTE;
    }

    seconds += (day - 1) * DAY * HOUR * MINUTE +
	    hour * HOUR * MINUTE + minute * MINUTE + second;
    return seconds;
}



/* ----------------------------------------------------------------
 * gmta.c, mod from gmt.c, to go before 1970.0 
 * 	a. porter, 91/09/18 ff.
 *	the orginal, gmt.c, was by GG Preckshot
 * 	
 * this should work by going back to the year in question,
 *  then going forward to the month, day, hour etc. in question
 *
 *
 * gmta() 
 *
 * Return double GMT seconds since 1970 (Unix time).  
 * 	This is civil time in the usual Unix format 
 *	of seconds since 00:00:00, Jan 1, 1970.
 * Fractional seconds are allowed.
 *
 * NOTE: month and day are in the ranges [1,12] and [1,31] respectively.
 *       for GMT(), year should be either [0,99] or [1970,2020(or so)].
 *	 here, for gmta(), year is absolute: it goes back to 1753.  
 * 	(in 1752, the year of change from Julian to Gregorian calendar in the UK, 
 *	(Wednesday September 2 is followed by Thursday September 14 (type <cal 1752>))
 * 	cf. _Explanatory Supplement to the Astronomical Epmemeris_ (1961), 
 *	ch. 14, p. 412f. for dates of conversion from Julian to Gregorian calendar
 *	in various places.  )  
 */



double
gmta( year, month, day, hour, minute, second )
    int		year, month, day, hour, minute;
    double	second;
{
    double	seconds = 0.0;
    int		yr,mo;


    if (year >= BASEYEAR) 
      return GMT( year, month, day, hour, minute, second ); 
    if (year < 1753)
    {
	printf(" you are out of the Gregorian calendar; Julian calendar not supported \n");
	printf(" returning value of 0.0 (=1970.0) \n"); 
	return 0;
    }

    /* count back by years from BASEYEAR (=1970.0), the unix zero of time: */ 
    for (yr=BASEYEAR-1; yr>=year; yr--)
    {
	seconds -= YEAR * DAY * HOUR * MINUTE;
	if ( leapyear( yr ) ) 
	{
	    seconds -= DAY * HOUR * MINUTE;
	}
    }
    /* we should now have unix time of the start of the year for the date sought */ 

    /* so count forward from jan 1.0 of the year to the date sought, as in GMT(): */
    if ( leapyear(year) )
    {
	for (mo=1; mo<month; mo++)
	    seconds += leapmonthdays[mo-1] * DAY * HOUR * MINUTE;
    }
    else
    {
	for (mo=1; mo<month; mo++)
	    seconds += monthdays[mo-1] * DAY * HOUR * MINUTE;
    }

    seconds += (day - 1) * DAY * HOUR * MINUTE +
	    hour * HOUR * MINUTE + minute * MINUTE + second;
    return seconds;
}


/* ----------------------------------------------------------------
 * leapyear() tells whether its argument is a leap-year or not 
 * ----------------------------------------------------------------
 */ 

int 
leapyear( year )
     int year; 
{
    if ( (year % 4) != 0)  	/* not divis by 4 --> not leap year */
      return 0;
    if (year < 1753) /* you are in the Julian calendar, which is simple */ 
    {
	return 1;
    }

    /* continuing in the Gregorian calendar, not simple: */ 
    if ( (year % 100) != 0)	/* divis by 4 but not 100 --> leap year */
      return 1; 
    if ( (year % 400) != 0)	/* 1700, 1800, 1900 are NOT leap years */
      return 0;
    else	
      return 1; 		/* but 2000, 2400, . . . ARE leap years */ 
}
	





/* ----------------------------------------------------------------
 * modified from jdnow() in noargs.c
 * no inputs
 * outputs julian day number for current time
 * knows only the gregorian calendar 
 */
  
double  
xjdnow()
{
	double xjd;		/* julian day */
	long utime;		/* seconds since 1970.0 */
	time_t t;		/* see Borland Manual, at localtime()  */ 
	struct tm *local;	/* see Borland Manual, at localtime()  */ 
	int isdst;		/* whether we are currently on daylight savings time */ 
 
 
	tzset();	 /* see e. g., Borland C manual, p. 534; 
			    this presumes <set TZ=PDT8PST> has set the environment */ 
 
	/* time() gets seconds since 1970.0 GMT; see Borland manual at time()  
	 * I don't know whether it corrects for leap-seconds; presumably not;
	 * as of 1993, there are about 30 of them
	 * see NASA NAIF-spice-lib file leapseco.ker or some such name.
	 */
	time( &utime ); 
 
	/* jd 2440587.5 is 1970.0, i. e., midnight 1969-12-31/1970-1-1,
	 * the start of the unix clock
	 */
	xjd = 2440587.5 + utime/86400.; 
 
	/* to find out whether we are on daylight savings time: */ 
	t = time(NULL);
	local = localtime( &t ); 
	isdst = local->tm_isdst; 
 
	/* (the Borland version handled this quite differently) */ 
	if (isdst)	/* correct for daylight savings */ 
	  xjd += (1.)/24.;	/* local daylight time */
	  
	return xjd; 
}
 


/* ----------------------------------------------------------------
 * these routines use UNIX system routines good only in approx interval (1902,2038)
 * they should not be used ... use routines above instead,
 * the methods of Fliegel and van Flandern
 * ----------------------------------------------------------------
 */ 

/* ----------------------------------------------------------------
 * calendarJulianToDate sets calendar date for input of Julian date 
 * inputs
 *	julian date
 * outputs
 *	ptr to gregorian calendar date string
 * side effects
 *	sets calendar date integers pointed to on input
 * method
 *	convert JD to unix-time, 
 *	and unix-time to Gregorian calendar date via unix system routines
 * cautions
 *	use the answer before you loose it . . . it is in static storage here
 * BUGS 
 *	bad before about 1902 or after about 2038 
 * ----------------------------------------------------------------
 */

char * 
calendarJulianToDate( jdate )
     double jdate;	/* double-float Julian date (1970.0 (0h UT Jan 1 1970) is JD 244 0587.5 */
{
    double j1970=2440587.5;	/* julian date of Jan 1 1970 0h UT (zero of Unix time) */
    double diffdays;		/* difference in days of jdate from unix zero */
    double seconds;		/* diffdays, converted to seconds = unix clock time */
    static char gregorian[15];	/* gregorian calendar date string */


    diffdays = jdate - j1970;
    seconds = diffdays * 86400.;
    
    strncpy( gregorian,  doubleToDate( seconds ), 13 );

    return gregorian;

}



/* ----------------------------------------------------------------
 * unixToCalendar converts double float time in unix seconds
 *	to character string, calendar format
 * inputs
 *	time in double-float, seconds since start of 1970 0h UT
 * outputs
 *	ptr to char string, calendar date
 * side effects
 *	sets date in internal static storage; 
 *	use before you call it again
 * cf. nice_date, in alSunMoon.c
 *	(argument there is simple integer, not double-float time) 
 * BUGS 
 *	bad before about 1902 or after about 2038 
 *	this seems to be a limit imposed by 2**32 size for integer seconds 
 *	in the call to ctime() 
 * ----------------------------------------------------------------
 */

char *
unixToCalendar( time )
     double time;		/* time in seconds since start of 1970 UT */
{
    long int clock;		/* the argument to the unix service-routine */
    char *udate;
    static char nice[30];	/* to hold the answer */
    int i;			/* loop control */

    /* the 8-hour fudge is for PST to GMT conversion: */
    clock = time + 8.*3600.;	/* convert double float to long integer */
    udate = ctime( &clock );	/* supposedly greenwich mean time */
    for (i=0; i<40; ++i) nice[i] = 0;
    for (i=0; i<21; ++i) nice[i] = udate[i+3];

    return nice; 
    
}


/* ----------------------------------------------------------------
 * doubleToDate returns a date string, gotten differently from doubloToCalendar
 * inputs
 *	time in double-float, seconds since start of 1970 0h UT
 * outputs
 *	character string
 *	(use it before you lose it)
 * side effects
 *	sets date in internal static storage
 * method
 *	calls unix routine gmtime instead of ctime;
 *	ctime() looks like it could get confused by time zone and daylight savings time
 * BUGS 
 *	bad before about 1902 or after about 2038 
 * ----------------------------------------------------------------
 */


char *
doubleToDate( time )
     double time;		/* time in seconds since start of 1970 UT */
{
    long int clock;		/* the argument to the unix service-routine */

    static char nice[100];	/* to hold the answer */
    int i;			/* loop control */


    struct tm *t;		/* parts of the time / date */

    clock = time;		/* convert double-float to long int */

    t = gmtime( &clock );		/* get time-struct */
    ++t->tm_mon;		/* months are 1-12 not 0-11 */

    for (i=0; i<100; ++i) nice[i] = '\0';

    /* now format the time in struct t to string nice: */
    (void) sprintf( nice, "  %d/%2d/%2d %2d:%2d:%2d", t->tm_year + 1900,
		  t->tm_mon, t->tm_mday, t->tm_hour, t->tm_min, t->tm_sec );

    return nice;	/* for now . . . better later */

}
