
/* ----------------------------------------------------------------
 * catToHeap gets a heap from a list of files in a cat-file 
 *      it assumes the default headerfile in the environment variable HEADERFILE
 *	it looks at the catalogFile, and decides which sort of format of files 
 *		the catalog refers to: JPL's 1987 format 
 *		(de/le-200, as shipped from JPL; no longer supported here), 
 *		or proprietary format 
 *			(translated into reader-friendly %e format with comments)
 * 	it then looks up the file containing the date you want 
 *	gets it, if it has not already been gotten 
 *	loads the desired TapeHeap from the file, and returns that 
 *
 *	this way the source and format of data are hidden from calling routines 
 *
 *
 * inputs
 *	jd for desired date  
 *	name of a catalog file 
 * outputs 
 *	ptr to a TapeHeap 
 * side effects
 *	fills the TapeHeap 
 * ----------------------------------------------------------------
 */ 

TapeHeap *
catToHeap( jd, catalogFile )
     double jd;		/* date of requested positions */ 
     char *catalogFile;	/* name of catalog file; 
			 *  full path to file that tells  
			 *  where the ephemeris files are 
			 */ 
{
    static TapeHeap *heap=NULL;		/* the ephemeris data heap */ 
    static FILE *fi=NULL;  		/* to open a data file with */ 
    static double tapeStart=1.e28;	/* what cat-file says 
					 * is start of ephemeris data file */
    static double tapeEnd=-1.e28;	/* enddate of file, acc. to cat-file */
    static char *lastcat=NULL; 		/* catalog file of last call */ 
    static char *headername;		/* path and name of tape header file */ 



    static char datafile[256];		/* name of data file from getTape */ 
    static int firsttime=1;
    static FILE *fl; 
    static TapeHeap saveheapBuf;
    static TapeHeap *saveheap=NULL; 


    if (firsttime)
    {
	fl = logfile(); 
	firsttime = 0; 
    }

    /* first we check to see if either TapeHeap on hand is right: */ 
    if (  !needHeap( catalogFile, heap, jd ) )
      return heap; 
    if (  !needHeap( catalogFile, saveheap, jd ) )
      return saveheap; 
    
    /* we (may) have an old TapeHeap,
     * and we are about to get a new one,
     * so save the old one in case we need it again soon ... 
     * (this is critical when apparentPlanetPosition()
     * 	  wants data back and forth from the start of one
     *    and the end of a previous data-file; without this save,
     * 	  the previous data file has to be traversed at every call)
     */ 
    if (  heap != NULL )
    {	/* copy heap to saveheap ... */ 
	saveheap = &saveheapBuf; 
	memcpy( saveheap, heap, sizeof( TapeHeap ) ); 
    }

    headername = getHeaderName(); 

    /* ---------------------------------------------------------------- */ 
    /* get the desired data-file */ 
    fi = getDataFile( jd, catalogFile, &tapeStart, &tapeEnd, datafile );	
    /* printf(" jd %12.2f file %s \n", jd, datafile ); */ 

    if (fi == NULL)
    {
	printf(" could not get an ephemeris file for date %12.2f ", jd );
	printf(" from getDataFile(); \n" );
	lastcat = NULL; 
	return (TapeHeap *) NULL; 
    }

    /* ---------------------------------------------------------------- */ 
    /* now get the heap from the data-file */ 
    if ( heap != NULL )
      heap = (TapeHeap *) NULL; 

    if ( binaryFormat( datafile ) )
      heap = GetHeapByFtell( headername, datafile, fi, jd ); 
    else if ( jplFormat( catalogFile ) )
    {
	/* heap = GetHeapFromJPL( fi, jd ); */
	printf(" you thought you needed to read an ephemeris file in \n");
	printf(" a very old JPL format -- the original (1988) de-200 format; \n");
	printf(" this format is no longer supported here ... calling exit(1) \n");
	exit(1);
    }
    else if ( odivFormat( catalogFile ) )
      heap = GetHeapFromOdiv( catalogFile, headername, fi, jd ); 
    else 
    {
	printf(" failing in catToHeap, jd %12.2f, cat file %s \n", 
	       jd, catalogFile ); 
	printf(" something is wrong with cat-file name format  \n"); 
	fprintf( fl, " failing in catToHeap, jd %12.2f, cat file %s \n", 
		jd, catalogFile ); 
	fprintf( fl, " something is wrong with cat-file format  \n"); 
	lastcat = NULL; 
	return (TapeHeap *) NULL; 
    }

    if (heap == NULL) 
    {
	printf(" in catToHeap, jd %12.2f, cat file %s \n", jd, catalogFile ); 
	printf(" GetHeapFromXXXX bombed \n" ); 
	fprintf( fl, " in catToHeap, jd %12.2f, cat file %s \n", 
		jd, catalogFile ); 
	fprintf( fl, " GetHeapFromXXXX bombed \n" ); 
	lastcat = NULL; 
	abort(); 	/* this is temporary (!!) */ 
	return (TapeHeap *) NULL; 
    }

    strncpy( heap->catfile, catalogFile, LONGlineLENGTH ); 
    lastcat = catalogFile; 

    /* printf(" catToHeap, jd %12.2f, catalogFile %s datafile %s \n", 
       jd, catalogFile, datafile ); */ 

    return heap; 

}
