/* ----------------------------------------------------------------
 * coortran.c collects coordinate transform routines
 * a. porter, 1994/10/03, from older code and other *.c files 
 *
 * precession, nutation, etc. are elsewhere;
 * these are for coord systems of one date or kind to those of another
 * ----------------------------------------------------------------
 */ 

#include <stdio.h>
#include <math.h>

#include "../include/ms.h"
#include "../include/matrix.h"
#include "../include/hnav.h"


extern char *malloc();
extern double normalize();

void almanacGreenNutation();


/* ----------------------------------------------------------------
 * almanacEquinox creates a matrix3d to transform FROM   J 2000.0    TO   date
 *	i. e., from equator & equinox coords of 2000.0 to equatorial coords of date
 * inputs
 * 	time
 *	matrix (i. e., a place to put the answer)
 * outputs
 *	none
 * side effects
 *	creates a matrix to transform positions from coords of 2000.0 to coords of date
 * sources 
 *	RM Green, p. 219, eqn. (9.23) for precessional angles
 *	RM Green, p. 221-222, eqn. (9.31), 
 *		for rotation matrix for precession from these angles 
 *	RM Green, p. 229, eqn. (9.52), corrected for what appears to be a typo,
 *		for nutation quantities,
 *	RM Green, p. 232, eqn. (9.62), for rotation matrix of nutation;
 *		eqn. (9.63) for their composition
 * suspicions
 *	the answers seem to be approximately right, 
 *	but their change with day seems to be jerky, not smooth, as in almanac
 *	I went to all double precision, but that does not seem to help . . . . 
 * ----------------------------------------------------------------
 */

void
almanacEquinox(utime, matrix)
     double utime;		/* time in seconds from 1970.0 */ 
     matrix3d matrix;		/* coords of J2000.0 pre-multiplied by this matrix 
				 * gives coords in system of date 
				 */ 

{
    /* precessional quantities: */
    double zeta; 	/* a precessional angle */
    double z;		/* a precessional angle */
    double theta;	/* a precessional angle */
    matrix3d precession; /* the rotation matrix for precession */
    matrix3d nutation;	/* the rotation matrix for nutation */

    /* times: */ 
    double time;		/* time from 1988.0 */
    double jdnow;		/* julian day of now-time */
    double t;			/* time now, in Julian centuriesb from J 2000.0 */
    double day;			/* time now, in days from start of year */
    double jd;			/* a julian date */

    /* miscellany */
    int i, j, k;	/* loop control */
    matrix3d unit;	/* unit matrix */


    /* ---------------------------------------------------------------- */
    time = utime - UNIXTIME_1988;
    jdnow = calendarUnixToJulian(utime);

    jd  = NINETEEN_EIGHTY_EIGHT;	/* JD for 1988.0, in astronomicalConstants.h */
    day = jd + time / 86400.;
    t = (day - 2451545.0 ) / 36525.;


    /* ---------------------------------------------------------------- */
    /* PRECESSION */
    /* rm_green_zeta and usnoPrecess both give parameters for nutation;
       choose one: */ 

    /* rm_green_zeta( utime, &zeta, &z, &theta); */ 
    usnoPrecess( 2451545.0, jdnow, &zeta, &z, &theta );

    precessionMatrix( zeta, z, theta, precession );

    /* ---------------------------------------------------------------- */
    /* NUTATION: */ 
    usnoNutate( jdnow, nutation );

    /* ---------------------------------------------------------------- */
    /* now multply precession and nutation for the answer . . . */
    /* matMult( nutation, precession, matrix); */
    /* we need double precision . . . (real.h may not be double): */ 
    for (i=0; i<3; ++i)
      for (j=0; j<3; ++j)
      {
	  matrix[i][j] = 0.;
	  for (k=0; k<3; ++k) 
	    matrix[i][j] += nutation[i][k] * precession[k][j];
      }


}



/* ----------------------------------------------------------------
 * almanacEcliptic prepares a matrix for transform from ecliptic to equatorial coords 
 * inputs
 *	time, double float, unix-time, seconds since 1970.0
 *	matrix3d; i. e., a ptr to a [3][3] array
 * outputs
 *	none
 * side effects
 * 	sets the matrix pointed to
 * source
 *	HM and USNO Nautical Almanac Offices, 
 *	Planetary and Lunar Coordinates for the Years 1984-2000
 *	Washington, USGPO, 1983
 *	(the blue book) 
 *	see p. 315 and 320
 * note, RM Green's formulas, pp. 229-230 of his book, 
 *	require lunar coords for their calculation
 * use
 *	this matrix rotates about the x-axis;
 *	i. e., it assumes you have already precessed the x-axis
 *	therefore, do the precession corrections FIRST 
 *	cf. p. 314-315 of <Planetary and Lunar Coordinates>
 * ----------------------------------------------------------------
 */

void
almanacEcliptic( utime, ecliptic )
     double utime;	/* unix-time, seconds since 1970.0 */
     matrix3d ecliptic;	/* a 3x3 array of doubles (the typedef is in matrix.h) */
{
    double obliquity;	/* about 23 degrees . . . */
    double t;		/* julian centuries from J 2000.0 */
    double jd;		/* julian day corresponding to utime */
    double delta_e;	/* nutation in obliquity (p. 320)  */
    double delta_psi;	/* nutation in longitude */
    double d;		/* jd - 2451545.0 */

    jd = calendarUnixToJulian( utime );
    d = jd - 2451545.0;			/* p. 320 */
    t = d   /  36525.;			/* p. 317 */

    GreenNutations( utime, &delta_psi, &delta_e, &obliquity );

    eclipticMatrix( obliquity, ecliptic );

}


/* the following routines came from celcoord.c: 
 * celcoord.c contains more routines for changes in coordinate systems
 *     from J2000.0 to date
 *     from ecliptic to equatorial 
 *	89/10/02 ff.
 *      94/06/07 reorganized somewhat 
 */



/* ----------------------------------------------------------------
 * eclipticMatrix constructs the rotation matrix for transformation
 *	from ecliptic to equatorial coordinates
 *	i. e., a rotation about the x-axis through the angle <obliquity>
 * inputs
 *	obliquity
 *	ptr to where to put the rotation matrix
 * outputs
 *	none
 * side effects
 *	puts matrix where you tell it to
 * ----------------------------------------------------------------
 */



void
eclipticMatrix( obliquity, ecliptic )
     double 	obliquity;		/* inclination of equator to ecliptic */
     matrix3d 	ecliptic;		/* rotation matrix from ecliptic to equatorial coordinates */
{
    EulerRot ecclipToEquat;	/* euler angles for the transform */

    ecclipToEquat.er_1 = 0.;
    ecclipToEquat.er_2 = obliquity;
    ecclipToEquat.er_3 = 0.; 
    matEulerToMat( &ecclipToEquat, ecliptic );

}


/* ----------------------------------------------------------------
 * creates same matrix, but from mean rather than true ecliptic
 * ----------------------------------------------------------------
 */

void
almanacMeanEcliptic( utime, ecliptic )
     double utime;	/* unix-time, seconds since 1970.0 */
     matrix3d ecliptic;	/* a 3x3 array of doubles (the typedef is in matrix.h) */
{
    double  obliquity;	/* about 23 degrees . . . */
    EulerRot ecclipToEquat;	/* euler angles for the transform */

    obliquity = almanacMeanObliquity( utime );

    ecclipToEquat.er_1 = 0.;
    ecclipToEquat.er_2 = obliquity;
    ecclipToEquat.er_3 = 0.; 
    matEulerToMat( &ecclipToEquat, ecliptic );

}


/* ----------------------------------------------------------------
 * toEcliptic transforms a vector from equatorial to eccliptic coords 
 * toEccliptic is miss-spelled, but see the #define in almanac.h 
 * inputs 
 *	jd
 *	ptr to vector to be rotated 
 * outputs 
 * 	none
 * side effects 
 *	rotates vector 
 *	saves its own copy of the rotation matrix 
 * ----------------------------------------------------------------
 */ 

void
toEccliptic( jd, vector )
     double 	jd;		/* julian date (determines obliquity of eccliptic */ 
     Coord3d	*vector; 
{
    static matrix3d rotamat;
    static jdlast=0.; 
    double obliquity; 
    matrix3d tempmat; 
    double T;			/* julian centuries from J 2000.0 */ 

    if (jd != jdlast)
    {	/* get new rotation matrix */ 
	/* cf. Almanac, p. B-18: */ 
	T = (jd - 2451545.0) / 36525.; 
	obliquity = 23.439291 - 0.0130042*T - 0.00000016*T*T + 0.000000504*T*T*T;
	obliquity *= TORADIANS; 
	obliquity *= -1.; 
	/* get matrix for transform from ecliptic to equatorial coords: */
	/* note the sign of the tilt has just been inverted */  
	eclipticMatrix( obliquity, rotamat );
	jdlast = jd; 
    }

    matMatVV( rotamat, vector, vector ); 

}





/* ----------------------------------------------------------------
 * toEquatorial transforms a vector from ecliptic to equatorial coords 
 * inputs 
 *	jd
 *	ptr to vector to be rotated 
 * outputs 
 * 	none
 * side effects 
 *	rotates vector 
 *	saves its own copy of the rotation matrix 
 * ----------------------------------------------------------------
 */ 

void
toEquatorial( jd, vector )
     double 	jd;		/* julian date (determines obliquity of eccliptic */ 
     Coord3d	*vector; 
{
    static matrix3d rotamat;
    static jdlast=0.; 
    double obliquity; 
    matrix3d tempmat; 
    double T;			/* julian centuries from J 2000.0 */ 

    if (jd != jdlast)
    {	/* get new rotation matrix */ 
	/* cf. Almanac, p. B-18: */ 
	T = (jd - 2451545.0) / 36525.; 
	obliquity = 23.439291 - 0.0130042*T - 0.00000016*T*T + 0.000000504*T*T*T;
	obliquity *= TORADIANS; 
	eclipticMatrix( obliquity, rotamat );

	jdlast = jd; 
    }

    matMatVV( rotamat, vector, vector ); 

}



/* this little test routine exercizes toEquatorial() */ 

int 
ecTest()
{
    Coord3d  vec; 

    printf("\n"); 
    vec.z = 1.; vec.x = vec.y = 0.; 
    dumpVec( &vec, "in eclip" ); 
    toEquatorial( 2451545.0, &vec ); 
    dumpVec( &vec, "to equat" ); 

    printf("\n"); 
    vec.z = 1.; vec.x = vec.y = 0.; 
    dumpVec( &vec, "in equat" ); 
    toEccliptic( 2451545.0, &vec ); 
    dumpVec( &vec, "to eclip" ); 

    printf("\n"); 
    vec.z = 0.; vec.x = 0.; vec.y = 1.; 
    dumpVec( &vec, "in eclip" ); 
    toEquatorial( 2451545.0, &vec ); 
    dumpVec( &vec, "to equat" ); 

    printf("\n"); 
    vec.z = 0.; vec.x = 0.; vec.y = 1.; 
    dumpVec( &vec, "in equat" ); 
    toEccliptic( 2451545.0, &vec ); 
    dumpVec( &vec, "to eclip" ); 

}




/* ----------------------------------------------------------------
 * almanacMeanObliquity returns the mean obliquity of equator to equinox
 * inputs
 *	unix time
 * outputs
 *	mean obliquity
 * side effects
 *	none
 * method
 *	series of RM Green, p. 220, eq. (9.25)
 * units
 *	seconds since 1970.0 in
 *	radians out
 * ----------------------------------------------------------------
 */

double 
almanacMeanObliquity( utime )
     double utime;	/* unix time */
{
    double obliquity;	/* obliquity of equator to equinox */
    double t;		/* julian centuries from J 2000.0 */
    double jd;		/* julian day corresponding to utime */
    double d;		/* jd - 2451545.0 */
    EulerRot ecclipToEquat;	/* euler angles for the transform */

    jd = calendarUnixToJulian( utime );
    d = jd - 2451545.0;			/* p. 320 */
    t = d   /  36525.;			/* p. 317 */

    /* for the constants, see RM Green, p. 220, eq. (9.25): */
    /* constants, almanac 1988, p. B18.  RM Green's are a little different */
    obliquity = ( 23.439291111 - 0.013004167*t - 1.6e-7*t*t + 5.04e-7*t*t*t);
    obliquity *= TORADIANS;

    return obliquity;

}


/* ----------------------------------------------------------------
 * equatorialPole( utime, pole )
 *	returns the pole of the mean equator 
 *	i. e., presumably changes with precession, but ignores nutation 
 * inputs
 *	unix-time
 *	ptr to Coord3d, where to put answer 
 * outputs
 *	none
 * side effects
 *	puts pole where you tell it to 
 * source
 *	C. A. Murray, Vectorial Astrometry, p. 107
 * ----------------------------------------------------------------
 */

void
equatorialPole( utime, pole )
     double 	utime;	/* unix time */
     Coord3d	*pole;	/* where to put the equatorial pole */
{
    double T;	/* Julian centuries from J 2000.0 */

    T = ( calendarUnixToJulian( utime ) - 2451545.0 ) / 36525.;

    pole->x = 0. + 0.009717173683*T - 0.000002068442*T*T - 0.000000963069*T*T*T;
    pole->y = 0.397777155932 + 0.   - 0.000118460841*T*T + 0.000000016167*T*T*T + 0.000000005881*T*T*T*T; 
    pole->z = 0.917482062070 + 0.   - 0.000000098874*T*T + 0.000000014897*T*T*T;

}



/* ----------------------------------------------------------------
 * eclipticPole( utime, pole )
 *	returns the pole of the ecliptic
 *	law of its change is unknown to me
 * inputs
 *	unix-time
 *	ptr to Coord3d, where to put answer 
 * outputs
 *	none
 * side effects
 *	puts pole where you tell it to 
 * source
 *	C. A. Murray, Vectorial Astrometry, p. 107
 * 	(a power series derived from something)
 * ----------------------------------------------------------------
 */

void
eclipticPole( utime, pole )
     double 	utime;	/* unix time */
     Coord3d	*pole;	/* where to put the ecliptic pole */
{
    double T;	/* Julian centuries from J 2000.0 */

    T = ( calendarUnixToJulian( utime ) - 2451545.0 ) / 36525.;

    pole->x = 0. + 0.000020350447*T + 0.000000942808*T*T + 0.000000000866*T*T*T;
    pole->y = 0. + 0.000226965993*T - 0.000000245249*T*T - 0.000000001665*T*T*T;
    pole->z = 1. + 0.            *T - 0.000000025964*T*T + 0.000000000036*T*T*T;

}




/* ----------------------------------------------------------------
 * cliptoclip2000 returns the transform matrix from ecliptic coords of date
 *	to ecliptic coords of J 2000.0
 *	the routine name has been condensed 
 *	("ecliptic to ecliptic of 2000" --> "cliptoclipt2000" )
 * inputs
 *	unix time
 *	transform matrix to be set
 * outputs
 *	none
 * side effects
 *	sets the matrix
 * sources, in the routines called for the two poles: 
 *	C A Murray's formulas for ecliptic and equator
 *	(Vectorial Astrometry, p. 107)
 * ----------------------------------------------------------------
 */

void 
cliptoclip2000( time, mat )
     double 	time;	/* unix time, seconds since 1970.0 */
     matrix3d	mat;	/* transform matrix from eclip. of date to eclip. of J 2000.0 */
{
    Coord3d	clippole;	/* ecliptic pole of date */
    Coord3d	quatpole;	/* equatorial pole of date */
    Coord3d	xaxis;		/* x axis (equinox) of date */ 
    matrix3d	xmat;		/* a temp storage place */ 

    eclipticPole(   time, &clippole );
    equatorialPole( time, &quatpole );
    GeoCrossProduct( &quatpole, &clippole, &xaxis );

    ZXtoMatrix( &clippole, &xaxis, xmat );
    matMatTranspose( xmat, mat ); 	/* added 92/01/29; makes little difference, but necessary */ 

}


/* ----------------------------------------------------------------
 * p1950to2000 precesses coordinates from B 1950.0 to J 2000.0
 *	i. e., from the FK4 system to the FK5 
 * inputs
 *	ptr to matrix, DOUBLE PRECISION
 * outputs
 *	none
 * side effects
 *	sets the matrix
 * warnings
 *	this is double precision, NOT matrix3d 
 *	i. e., truncate it yourself, or use fk4to5() 
 * source
 * 	Jay H. Lieske, Astron. & Astroph. 73 (1979) 282,
 *	"Precession Matrix Based on IAU (1976) System of Astronomical Constants
 * ----------------------------------------------------------------
 */

void
p1950to2000(mat)
     double mat[3][3]; 
{
    mat[0][0] =   0.9999257079523629;
    mat[0][1] = - 0.0111789381377700;
    mat[0][2] = - 0.0048590038153592;

    mat[1][0] =   0.0111789381264276;
    mat[1][1] =   0.9999375133499888;
    mat[1][2] = - 0.0000271625947142;

    mat[2][0] =   0.0048590038414544;
    mat[2][1] = - 0.0000271579262585;
    mat[2][2] =   0.9999881946023742;

}

void
fk4to5( mat )
     matrix3d mat;
{
    double dmat[3][3];
    int i, j;

    p1950to2000(dmat);
    for (i=0; i<3; ++i) 
      for (j=0; j<3; ++j) 
	mat[i][j] = dmat[i][j];

}
    

/* ----------------------------------------------------------------
 * cliptoquat transforms a vector from ecliptic to equatorial coords
 *	in the reference systems of J 2000.0
 *	i. e., mean equator and equinox and ecliptic of J 2000.0
 * inputs
 *	ptr to a Coord3d vector 
 * outputs
 * 	none
 * side effects
 *	changes the vector from ecliptic to equatorial coords
 * ----------------------------------------------------------------
 */

void 
cliptoquat( v )
     Coord3d *v;	/* the vector to be rotated */
{
    static matrix3d 	ecliptic2000; 	/* the rotation matrix for epoch J 2000.0 */ 
    static int 		ready=0;	/* whether we have prepared ecliptic yet or not */
    Coord3d		temp;		/* temp storage */

    if (!ready)
    {
	eclipticMatrix( 23.43929111*TORADIANS, ecliptic2000 );
	ready = 1;
    }

    matMatVV( ecliptic2000, v, &temp );
    GeoScalarMultVector( 1., &temp, v );

}


/* ----------------------------------------------------------------
 * quatPos gives planet position and velocity
 * 	in equatorial coords of J 2000.0 in AU and AU/day
 * inputs
 *	orbit, referred to mean ECLIPTIC and equinox of J 2000.0
 *	time, seconds from 1970.0 
 *	ptr to position, to be set 
 *	ptr to velocity, to be set  
 * outputs
 *	none
 * side effects
 *	sets position and velocity
 * called-from locations
 *      seem to be in apparent.c and fast.c 
 * about the zero of time: 
 *	it could be arbitary, here it is 1970.0,
 *	but I suppose it is really defined by the zero
 *	to which the orbit o_periTime member is referenced ... 
 * 	(if you change it one place, it will have to be changed in others)
 * ----------------------------------------------------------------
 */

void
quatPos( orb, time, position, velocity )
     Orbit 	*orb;		/* the orbit for which position is sought */ 
     double	time;		/* unix time in seconds */
     Coord3d	*position;	/* position sought, to be in AU */
     Coord3d	*velocity;	/* velocity sought, to be in AU/day */
{ 
    double 	true_a;		/* true anomaly */
    
    /* get position and velocity in MKS units, ecliptic coords: */
    true_a = navTrueAnomaly( orb, time ); 

    navAnomalyToPosition( orb, true_a, position );
    navVelocity( orb, true_a, velocity );

    /* change units to AU, AU/day: */
    GeoScalarMultVector( 1./AU, position, position );
    GeoScalarMultVector( 86400./AU, velocity, velocity ); 
				   
    /* change to equatorial coords: */
    cliptoquat( position );
    cliptoquat( velocity ); 

}





/* ----------------------------------------------------------------
 * meanObliquity returns the mean obliquity of the eccliptic to the equator of date 
 * input
 *	unix time 
 * output
 * 	mean obliquity of etc., units are radians
 * side effects
 *	none
 * source
 *	almanac, p. B-18
 * ----------------------------------------------------------------
 */ 


double meanObliquity( utime ) 
     double utime;		/* unix time, SI seconds since 1970.0 */ 
{
    double jd; 			/* julian date */
    double T;			/* julian centuries from J 2000.0 */ 
    double obliquity;		/* mean obliquity of eccliptic to equator of date */

    jd = calendarUnixToJulian( utime );
    T = ( jd - 2451545.0) / 36525.;

    obliquity = 23.439291 - 0.0130042*T - 0.00000016*T*T - 0.000000504*T*T*T;
    
    return obliquity*TORADIANS;

}








