/* exorbs.c does keplerian orbits from the _Explanatory Supplement
 *     to the Almanac_; they are exercized in ../progs/fast.c 
 *	and (if you specify orbits on the command line) ../progs/planets.c
 */ 

#define LANGUAGE_C

#include <stdio.h>
#include <math.h>
#include <time.h>
 
#include "../include/ms.h"
#include "../include/matrix.h" 
#include "../include/hnav.h" 

 
void ExplanatorySupplementOrbits();
double  jdnow();
 
/* this is "public" storage */
 
Orbit almanacOrbit_Mercury;
Orbit almanacOrbit_Venus;
Orbit almanacOrbit_Earth;
Orbit almanacOrbit_Mars;
Orbit almanacOrbit_Jupiter;
Orbit almanacOrbit_Saturn;
Orbit almanacOrbit_Uranus;
Orbit almanacOrbit_Neptune;
Orbit almanacOrbit_Pluto;
Orbit almanacOrbit_PlanetX;

Orbit *planet[10] = {
 &almanacOrbit_Mercury,
 &almanacOrbit_Venus,
 &almanacOrbit_Earth,
 &almanacOrbit_Mars,
 &almanacOrbit_Jupiter,
 &almanacOrbit_Saturn,
 &almanacOrbit_Uranus,
 &almanacOrbit_Neptune,
 &almanacOrbit_Pluto,
 &almanacOrbit_PlanetX
};


char  *planetName[10] = {
    "Mercury",
    "Venus  ",
    "Earth  ",
    "Mars   ",
    "Jupiter",
    "Saturn ",
    "Uranus ",
    "Neptune",
    "Pluto  ",
    "PlanetX"
 };

 
 
 

/* ----------------------------------------------------------------
 * orbits of the planets:
 * input
 *	time, SI seconds since 1988 January 01 0h UT1
 * outputs
 *	none
 * side effects
 *	puts elements in almanac planet orbits,
 *	for which storage is declared herewith
 *
 * source
 * 	Astronomical Almanac 1988, p. E3
 *	these are the osculating elements for JD 244 7400.0 (1988/08/26, 12h UT1)
 *				     and  for JD 244 7200.0 (1988/02/08, 12h UT1)
 *	the routine interpolates between (extrapolates beyond) these dates for the osculating elements
 *
 * notes
 *	these are orbital elements referred to mean ecliptic and equinox of J2000.0
 *	you must transform to geocentric and/or equatorial coords if that's what you want 
 *
 * ----------------------------------------------------------------
 */
 
 
 
 

/* ----------------------------------------------------------------
 * from Explanatory Supplement to the Astronomical Almanac, 1992, p. 316
 * coords = mean ecliptic and equinox of J 2000.0
 * units: degrees, plus arc-seconds of change per century 
 * these are linear approximations; good only from 1800-2050
 * ----------------------------------------------------------------
 */
 
void
ExplanatorySupplementOrbits( jd )
     double jd;	  	/* Julian Date */ 
{
  double T;	      /* time in centuries from J2000.0 */ 
  double mass;		/* temp for sun mass */
  double ptime;	/* temp for zero perigee time */
  double ecc;		/* temp for eccentricity */
  double axis;		/*      for semi-major axis */
  double node;		/*      for longitude of ascending node */
  double incl;		/*      for inclination to eccliptic */
  double pgee;		/*	for argument of perigee, measured from ascending node */
  double refTime;	/* 	for reference time */
  double refAnomaly;	/* 	for reference anomaly */
  double tfac;		/*    time factor for linear changes */ 
  double tfacAng;	/*    time factor with conversion from seconds to degrees */
  double temp;
  
    refTime	= calendarJulianToUnix( jd ); 
    
    /* mass of the central body: */
    mass 	= 1.32712438e+20/BIG_G;	/* mks units; Expl Suppl, p. 696 */
    /* NOTE: perigee time may be supplied as zero, 
     * it will be reset to conform to refAnomaly 
     */
    ptime 	= 0.;			/* will be reset, in navOrbitSet() */
 
    /*  NOTE:  Expl Suppl, p. 316, gives perigee+node, and ref.Anomaly+node+perigee;
     * 		the notation is explained in RM Green, p. 164.
     *	these must be compensated for, 
     *		in addition to the conversion from degrees to radians
     */
 
    T = (jd - 2451545.0) / 36525.; 
    tfac = T;
    tfacAng = tfac / 3600.; 
    /* the time-linear rates for angles are in arc-seconds per century */ 
    
  /* units of orbital elements:
   * axis: AU
   * eccentricity: dimensionless
   * inclination: degrees
   * node: degrees
   * pgee, perigee: degrees; includes argument of ascending node 
   * refAnomaly: degrees; includes node and perigee
   */ 

    /* mercury */
    axis	= 0.38709893 + 0.00000066*tfac;		
    ecc 	= 0.20563069 + 0.00002527*tfac;		
    incl	= 7.00487 - 23.51*tfacAng;		
    node	= 48.33167 - 446.30*tfacAng;		
    pgee	= 77.45645 - 573.57*tfacAng;		
    refAnomaly  = 252.25084 + (261628.29 + 415.*3600.*360.)*tfacAng;
 
    orbitUnits( &axis, &node, &incl, &pgee, &refAnomaly);
    navOrbitSet( &almanacOrbit_Mercury, mass, ecc, ptime, axis, node, incl, pgee, refAnomaly, refTime );
 
    /* Venus */
    axis	= 0.72333199 + 0.00000092*tfac;		
    ecc 	= 0.00677323 - 0.00004938*tfac;		
    incl	= 3.39471 - 2.86*tfacAng;		
    node	= 76.68069 - 996.89*tfacAng;		
    pgee	= 131.53298 - 108.80*tfacAng;		
    refAnomaly  = 181.97973 + (712136.06 + 162.*3600.*360.)*tfacAng;
 
    orbitUnits( &axis, &node, &incl, &pgee, &refAnomaly);
    navOrbitSet(&almanacOrbit_Venus,   mass, ecc, ptime, axis, node, incl, pgee, refAnomaly, refTime );
 
    /* Earth */
    axis	= 1.00000011 -0.00000005*tfac;		
    ecc 	= 0.01671022 - 0.00003804*tfac;		
    incl	= 0.00005 - 46.94*tfacAng;		
    node	= -11.26064 - 18288.25*tfacAng;		
    pgee	= 102.94719 + 1198.28*tfacAng;		
    refAnomaly  = 100.46435 + (1293740.63 + 99.*3600.*360.)*tfacAng;
 
    refAnomaly  = 100.46435;
    temp = (1293740.63 + 99.*3600.*360.)*tfacAng;  
    refAnomaly  += temp;
 
    orbitUnits( &axis, &node, &incl, &pgee, &refAnomaly);
    navOrbitSet(&almanacOrbit_Earth,   mass, ecc, ptime, axis, node, incl, pgee, refAnomaly, refTime );
 
    /* Mars */
    axis	= 1.52366231 - 0.00007221*tfac;		
    ecc 	= 0.09341233 + 0.00011902*tfac;		
    incl	= 1.85061 - 25.47*tfacAng;		
    node	= 49.57854 - 1020.19*tfacAng;		
    pgee	= 336.04084 - 1560.78*tfacAng;		
    refAnomaly  = 355.45332 + (217103.78 + 53.*3600.*360.)*tfacAng;
 
    orbitUnits( &axis, &node, &incl, &pgee, &refAnomaly);
    navOrbitSet(&almanacOrbit_Mars,    mass, ecc, ptime, axis, node, incl, pgee, refAnomaly, refTime );
 
    /* Jupiter */
    axis	= 5.20336301 + 0.00060737*tfac;		
    ecc 	= 0.04839266 - 0.00012880*tfac;		
    incl	= 1.30530 -  4.15*tfacAng;		
    node	= 100.55615 + 1217.17*tfacAng;		
    pgee	= 14.75385 - 839.93*tfacAng;		
    refAnomaly  =  34.40438 + (557078.35 +   8.*3600.*360.)*tfacAng;  
 
    orbitUnits( &axis, &node, &incl, &pgee, &refAnomaly);
    navOrbitSet(&almanacOrbit_Jupiter, mass, ecc, ptime, axis, node, incl, pgee, refAnomaly, refTime );
 
    /* Saturn */
    axis	= 9.53707032 - 0.00301530*tfac;		
    ecc 	= 0.05415060 - 0.00036762*tfac;		
    incl	= 2.48446 +  6.11*tfacAng;		
    node	= 113.71504 - 1591.05*tfacAng;		
    pgee	= 92.43194 - 1948.89*tfacAng;		
    refAnomaly  =  49.94432 + (513052.95 +   3.*3600.*360.)*tfacAng;
 
    orbitUnits( &axis, &node, &incl, &pgee, &refAnomaly);
    navOrbitSet(&almanacOrbit_Saturn,  mass, ecc, ptime, axis, node, incl, pgee, refAnomaly, refTime );
 
    /* Uranus */
    axis	= 19.19126393 + 0.00152025*tfac;	
    ecc 	= 0.04716771 - 0.00019150*tfac;		
    incl	= 0.76986 -  2.09*tfacAng;		
    node	= 74.22988 - 1681.40*tfacAng;		
    pgee	= 170.96424 - 1312.56*tfacAng;		
    refAnomaly  = 313.23218 + (246547.79 +   1.*3600.*360.)*tfacAng;
 
    orbitUnits( &axis, &node, &incl, &pgee, &refAnomaly);
    navOrbitSet(&almanacOrbit_Uranus,  mass, ecc, ptime, axis, node, incl, pgee, refAnomaly, refTime );
 
    /* Neptune */
    axis	= 30.06896348 - 0.00125196*tfac;	
    ecc 	= 0.00858587 + 0.00002514*tfac;		
    incl	= 1.76917 -  3.64*tfacAng;		
    node	= 131.72169 - 151.25*tfacAng;		
    pgee	= 44.97135 - 844.43*tfacAng;		
    refAnomaly  = 304.88003 + (786449.21)*tfacAng;  
 
    orbitUnits( &axis, &node, &incl, &pgee, &refAnomaly);
    navOrbitSet(&almanacOrbit_Neptune, mass, ecc, ptime, axis, node, incl, pgee, refAnomaly, refTime );
 
    /* Pluto */
    axis	= 39.48168677 - 0.00076912*tfac;	
    ecc 	= 0.24880766 + 0.00006465*tfac;		
    incl	= 17.14175 + 11.07*tfacAng;		
    node	= 110.30347 -  37.33*tfacAng;		
    pgee	= 224.06676 - 132.25*tfacAng;		
    refAnomaly  = 238.92881 + (522747.90)*tfacAng;  
 
    orbitUnits( &axis, &node, &incl, &pgee, &refAnomaly);
    navOrbitSet(&almanacOrbit_Pluto,   mass, ecc, ptime, axis, node, incl, pgee, refAnomaly, refTime );
 
}

/* ----------------------------------------------------------------
 * orbitUnits converts from ideosyncratic units and conventions of the Almanac 
 *		to reasonable units
 * inputs
 *	pointers to semi-major axis, node, inclination, perigee, reference anomaly
 * outputs
 *	none
 * side effects
 *	converts axis from AU to meters
 *	converts angles from degrees to radians
 *	removes node from perigee, and node+perigee from reference anomaly
 *	
 * ----------------------------------------------------------------
 */
 
void
orbitUnits( axis, node, incl, pgee, refAnomaly)
     double *axis;		/* semi-major axis, in AU */
     double *node;		/* longitude of the ascending node */
     double *incl;		/* orbital inclination to the eccliptic */
     double *pgee;		/* argument of the perigee */
     double *refAnomaly;		/* reference mean anomaly */
{
    *axis	*= AU; 
 
    *node	*= TORADIANS;
    *incl	*= TORADIANS;
    *pgee	*= TORADIANS;
    *refAnomaly  *= TORADIANS;
    *refAnomaly = normalize( *refAnomaly, 0., PI2 ); 
    *pgee	-= *node;
    *refAnomaly  -= *pgee + *node; 	/* note order of compensations matters here !!!! */
 
    (void) axis;    /* this is so the compiler won't complain . . . */ 
}
 

/* ----------------------------------------------------------------
 * jdnow returns the double-float julian day number for the current local time
 * inputs
 *    none
 * outputs
 *    double-float julian day number
 * side effects
 *    none
 *
 * the routines called are documented in the Borland C manuals;
 * and possibly in the C manuals of other compilers.
 * they follow approximately the UNIX conventions.
 *
 * this routine (see obsoleted code below) didn't correct for daylight savings;
 * now it just calls xjdnow() ... which does.
 */
 
 
 
double  
jdnow()
{
	double xjd;		/* julian day */
	long utime;		/* seconds since 1970.0 */
	time_t t;		/* see Borland Manual, at localtime()  */ 
	struct tm *local;	/* see Borland Manual, at localtime()  */ 
	int isdst;		/* whether we are currently on daylight savings time */ 
	long jd; 
 
	return xjdnow(); 	/* this is in calendar.c */ 

#ifdef notused 
	/* this is old code.  It worked, once. */ 

	tzset();	 /* see Borland C manual, p. 534; this presumes <set TZ=PDT8PST> has set the environment */ 
 
	/* time() gets seconds since 1970.0 GMT; see Borland manual at time()  
	 * I don't know whether it corrects for leap-seconds; presumably not;
	 * as of 1993, there are about 30 of them
	 * see NASA NAIF-spice-lib file leapseco.ker or some such name.
	 */
	time( &utime ); 
 
	/* jd 2440587.5 is 1970.0, i. e., midnight 1969-12-31/1970-1-1,
	 * the start of the unix clock
	 */
	xjd = 2440587.5 + utime/86400.; 
	  
	return xjd; 
#endif notused

}
 
