
/* ----------------------------------------------------------------
 * findheap.c: find a heap 
 *	routines to find a tape-heap (i. e., a 32-day data-record), 
 *	from files of ODiv format, in binary or ascii, 
 * 		by a. porter, 91/12/01  ff.
 * ----------------------------------------------------------------
 */


#define LANGUAGE_C
#include <stdio.h>
#include <stdlib.h>
#include <ctype.h>
#include <math.h>
#include <string.h>

/* link with nrutil.o, in same directory */
#include "../include/nrutil.h"

#include "../include/ms.h" 
#include "../include/matrix.h"
#include "../include/hnav.h"


extern double normalize();



/* ----------------------------------------------------------------
 * this was used to set up a file to log diagnostic info to ... 
 * is used as a place to log certain failures;
 * search on 'fl' to see where 
 * 94/12/05 
 * ----------------------------------------------------------------
 */ 



FILE *
logfile()
{
    static FILE *fl;
    static int firsttime=1;

    if (firsttime)
    {
    	fl = fopen( "/dev/null", "w" ); 
	if ( fl == NULL )
	{
	    printf(" could not open log file \n" );
	    exit(1);
	}
	firsttime = 0;
    }
    return fl;
}
    



/* ----------------------------------------------------------------
 * returns the value of the environmental valiable CATALOGFILE,
 * 	with path-name of the data-directory prepended to it, 
 *	or else a default value, if CATALOGFILE is not set.
 * ----------------------------------------------------------------
 */ 

char *
getEphemerisCatalog()
{
    static char *catfilename;	/* catalog file name for jpl ephemeris */ 
    static char	tempname[256];	/* a temp pointer */

    static char catbuf[512];		/* where to put it */ 
    char *ephloc;			/* to hold value of env variable "ephloc" */ 
    char *tempstring; 
    char *catfile; 


    catfilename = "cat-odiv-ascii";  
    catfile = getenv( "CATALOGFILE" );
    if (catfile == NULL)
    {
	printf(" could not get environment variable CATALOGFILE; \n");
	printf(" it should name the catalog file in directory at EPHLOC \n");
	printf(" it is simplest if you set the CATALOGFILE variable and re-run \n" );
	exit( 1 ); 
    }


    strcpy( tempname, catfile );
    if ( strlen(tempname) > 0 )
    {
	catfilename = tempname; 
    }

    ephloc = getenv( "EPHLOC" ); 
    if (ephloc == NULL)
    {
	printf(" could not get environment variable EPHLOC; \n");
	printf(" it should tell the directory containing translated JPL ephemeris data \n"); 
	printf(" it is simplest if you set the EPHLOC variable and re-run \n" );
	exit( 1 ); 
    }

    strncpy( catbuf, ephloc, 128 );
    strcat( catbuf, "/" ); 
    strcat( catbuf, catfilename ); 
    catfilename = catbuf; 
    
    return catfilename; 
}
 
 

/* ----------------------------------------------------------------
 * getHeaderName returns the header file name specified 
 * 	by environment variable HEADERFILE, if it is not null,
 *	or a default name, if it is null.
 * ----------------------------------------------------------------
 */  

char *
getHeaderName()
{
    static char *headername;		/* to hold the path and name of the tape header file */ 
    static char headerbuf[128];		/* where to put it */ 
    char *ephloc;			/* to hold value of env variable "ephloc" */ 
    char *tempstring; 
    char *headerfile; 

    ephloc = getenv( "EPHLOC" ); 
    if (ephloc == NULL)
    {
	printf(" could not get environment variable EPHLOC; \n");
	printf(" it should tell the directory containing translated JPL ephemeris data \n"); 
	printf(" it is simplest if you set the EPHLOC variable and re-run \n" );
	exit( 1 ); 
    }

    strncpy( headerbuf, ephloc, 100 );
    strcat( headerbuf, "/" ); 
    headerfile = "header"; 
    tempstring = getenv( "HEADERFILE" ); 
    if ( tempstring == NULL)
    {
	printf(" could not get environment variable HEADERFILE; \n");
	printf(" it should name the translated headerfile in the directory at EPHLOC \n");
	printf(" it is simplest if you set the HEADERFILE variable and re-run \n"); 
	exit( 1 ); 
    }
    else
      headerfile = tempstring; 
    strcat( headerbuf, headerfile ); 
    headername = headerbuf; 

    return headername; 

}


/* ----------------------------------------------------------------
 * these routines tell whether a filename indicates JPL or ODiv format 
 * (some possibilities are no longer used ...)
 * jpl format files have ".jpl" in their names
 * odiv ascii format files have ".odiv" or "-odiv" in their names
 * odiv binary format files have ".bin" in their names 
 * ----------------------------------------------------------------
 */ 


int 
jplFormat( filename )
     char *filename;	/* the catalog file to be interrogated */ 
{
    /* the "." is crucial: "jpl" in pathname does NOT count, ".jpl" does */ 
    if ( strstr( filename, ".jpl")  != NULL )
    {
	return 1;
    }
    else return 0; 
}

int 
odivFormat( filename )
     char *filename;	/* the catalog file to be interrogated */ 
{
    /* both ".odiv" and "-odiv" count */ 
    if ( strstr( filename, "odiv" ) != NULL )
	return 1; 

    else return 0; 
}

int 
binaryFormat( datafile )
     char *datafile;
{
    if ( strstr(datafile, ".bin") != NULL )
      return 1; 
    else 
      return 0;   
}




/* ----------------------------------------------------------------
 * needHeap tells you whether the heap you have is not what you want 
 * inputs
 * 	catalogfile: the name of the cat file from which a heap is desired
 *	heap:	pointer to the heap you have, the heap in question 
 * 			(NULL if you don't have one)
 *	jd: the date for which a heap is sought
 * outputs
 *	1 for yes, you need a new heap; 0 for no, the one you have is good 
 * side effects
 *	none
 * ----------------------------------------------------------------
 */

int 
needHeap( catfile, heap, jd )
     char *catfile;
     TapeHeap *heap;
     double jd;
{

    if ( heap == NULL )
      return 1; 

    if ( jd < heap->jdStart  ||  jd > heap->jdEnd )
      return 1; 

    if ( strncmp( catfile, heap->catfile, LONGlineLENGTH ) != 0)
      return 1;

    return 0; 

}




/* ----------------------------------------------------------------
 * catHeadToHeap gets a heap for date per catalog and headerfile 
 *      in its arg list; 
 *	it looks at the catalogFile, and decides which sort of format of files 
 *		the catalog refers to: JPL's 1987 format 
 *		(de/le-200, as shipped from JPL; no longer supported here), 
 *		or proprietary format 
 *			(translated into reader-friendly %e format with comments)
 * 	it then looks up the file containing the date you want 
 *	gets it, if it has not already been gotten 
 *	loads the desired TapeHeap from the file, and returns that 
 *
 *	this way the source and format of data are hidden from calling routines 
 *
 *
 * inputs
 *	jd for desired date  
 *	name of a catalog file 
 *      name of headerfile 
 * outputs 
 *	ptr to a TapeHeap 
 * side effects
 *	fills the TapeHeap 
 * ----------------------------------------------------------------
 */ 

TapeHeap *
catHeadToHeap( jd, catalogFile, headername )
     double jd;		/* date of requested positions */ 
     char *catalogFile;	/* name of catalog file; 
			 *  full path to file that tells  
			 *  where the ephemeris files are 
			 */ 
     char *headername;  /* the name of the headerfile */ 
{
    static TapeHeap *heap=NULL;		/* the ephemeris data heap */ 
    static FILE *fi=NULL;  		/* to open a data file with */ 
    static double tapeStart=1.e28;	/* what cat-file says 
					 * is start of ephemeris data file */
    static double tapeEnd=-1.e28;	/* enddate of file, acc. to cat-file */
    static char *lastcat=NULL; 		/* catalog file of last call */ 

    static char datafile[256];		/* name of data file from getTape */ 
    static int firsttime=1;
    static FILE *fl; 
    static TapeHeap saveheapBuf;
    static TapeHeap *saveheap=NULL; 


    if (firsttime)
    {
	fl = logfile(); 
	firsttime = 0; 
    }

    /* first we check to see if either TapeHeap on hand is right: */ 
    if (  !needHeap( catalogFile, heap, jd ) )
      return heap; 
    if (  !needHeap( catalogFile, saveheap, jd ) )
      return saveheap; 
    
    /* we (may) have an old TapeHeap,
     * and we are about to get a new one,
     * so save the old one in case we need it again soon ... 
     * (this is critical when apparentPlanetPosition()
     * 	  wants data back and forth from the start of one
     *    and the end of a previous data-file; without this save,
     * 	  the previous data file has to be traversed at every call)
     */ 
    if (  heap != NULL )
    {	/* copy heap to saveheap ... */ 
	saveheap = &saveheapBuf; 
	memcpy( saveheap, heap, sizeof( TapeHeap ) ); 
    }

    /* ---------------------------------------------------------------- */ 
    /* get the desired data-file */ 
    fi = getDataFile( jd, catalogFile, &tapeStart, &tapeEnd, datafile );	
    /* printf(" jd %12.2f file %s \n", jd, datafile ); */ 

    if (fi == NULL)
    {
	printf(" could not get an ephemeris file for date %12.2f ", jd );
	printf(" from getDataFile(); \n" );
	lastcat = NULL; 
	return (TapeHeap *) NULL; 
    }

    /* ---------------------------------------------------------------- */ 
    /* now get the heap from the data-file */ 
    if ( heap != NULL )
      heap = (TapeHeap *) NULL; 

    if ( binaryFormat( datafile ) )
      heap = GetHeapByFtell( headername, datafile, fi, jd ); 
    else if ( jplFormat( catalogFile ) )
    {
	/* heap = GetHeapFromJPL( fi, jd ); */
	printf(" you thought you needed to read an ephemeris file in \n");
	printf(" a very old JPL format -- the original (1988) de-200 format; \n");
	printf(" this format is no longer supported here ... calling exit(1) \n");
	exit(1);
    }
    else if ( odivFormat( catalogFile ) )
      heap = GetHeapFromOdiv( catalogFile, headername, fi, jd ); 
    else 
    {
	printf(" failing in catHeadToHeap, jd %12.2f, cat file %s \n", 
	       jd, catalogFile ); 
	printf(" something is wrong with cat-file name format  \n"); 
	fprintf( fl, " failing in catHeadToHeap, jd %12.2f, cat file %s \n", 
		jd, catalogFile ); 
	fprintf( fl, " something is wrong with cat-file format  \n"); 
	lastcat = NULL; 
	return (TapeHeap *) NULL; 
    }

    if (heap == NULL) 
    {
	printf(" in catToHeap, jd %12.2f, cat file %s \n", jd, catalogFile ); 
	printf(" GetHeapFromXXXX bombed \n" ); 
	fprintf( fl, " in catHeadToHeap, jd %12.2f, cat file %s \n", 
		jd, catalogFile ); 
	fprintf( fl, " GetHeapFromXXXX bombed \n" ); 
	lastcat = NULL; 
	abort(); 	/* this is temporary (!!) */ 
	return (TapeHeap *) NULL; 
    }

    strncpy( heap->catfile, catalogFile, LONGlineLENGTH ); 
    lastcat = catalogFile; 

    /* printf(" catHeadToHeap, jd %12.2f, catalogFile %s datafile %s \n", 
       jd, catalogFile, datafile ); */ 

    return heap; 

}


/* ----------------------------------------------------------------
 * catToHeap gets a heap from a list of files in a cat-file 
 *      it gets a header-file name and calls catHeadToHeap()
 *      to see how things work, look at that routine 
 *
 * inputs
 *	jd for desired date  
 *	name of a catalog file 
 * outputs 
 *	ptr to a TapeHeap 
 * side effects
 *	the called routine, catHeadToHeap(), fills the TapeHeap 
 * ----------------------------------------------------------------
 */ 

TapeHeap *
catToHeap( jd, catalogFile )
     double jd;		/* date of requested positions */ 
     char *catalogFile;	/* name of catalog file; 
			 *  full path to file that tells  
			 *  where the ephemeris files are 
			 */ 
{
    static char *headername;		/* path and name of tape header file */ 
    static TapeHeap *heap=NULL; 

    headername = getHeaderName(); 
    heap = catHeadToHeap( jd, catalogFile, headername ); 

    return heap; 

}


/* ----------------------------------------------------------------
 * getDataFile finds the ephemeris file for the date you want 
 * inputs
 *	julian date 
 * 	name of a catalog of ephemeris files 
 *		for format, see read stmts below 
 * outputs 
 *	a FILE descriptor, or NULL, if failure 
 * side effects 
 * 	sets tapeStart, tapeEnd 
 *      copies the file-name to the char-buffer pointed to by filename
 *          (when last looked at, the buffer was 256 chars long) 
 * this works for BOTH tape-files in JPL and Odiv formats:
 *	the catalog files have the same format 
 *	(the "jpl" format of ca. 1987-1988 is no longer supported)
 * NOTE
 *	this routine _does_ return the file name; 
 *	an earlier version ( getTape() ) did not. 
 * 	If you call it, you will later be able to get faster access
 * ----------------------------------------------------------------
 */ 

FILE *
getDataFile( jd, catalogFile, tapeStart, tapeEnd, filename )
     double jd;		/* jd for which ephemeris file is sought */ 
     char *catalogFile;	/* a file with list of ephemeris files in it */ 
     double *tapeStart;	/* read from the catalogFile, to be set by getTape */ 
     double *tapeEnd;	/* ditto */ 
     char   *filename;	/* char buffer to hold the filename that is sought;
			   we will write to this buffer */ 
{
    double 	jdStart;	/* a julian date */ 
    double 	jdEnd;		/* a julian date */ 

    static FILE  *fcat=NULL;	/* ptr to the catalog file */

    static FILE  *fi=NULL;	/* ptr to the file sought */ 
    static FILE  *oldfile=NULL;	/* the last data file we found */ 

    char       *line; 		/* a line read from catalogFile */ 
    int		count;		/* count of assignments made by fscanf()  */ 
    char 	buf[256]; 
    static char oldcatfile[128];	/* copy of last catalogFile */ 
    static char ephloc[128];	/* directory location of ephemeris data files */ 
    char	*tempname; 
    char	tempbuf[256]; 
    int		firsttime=1; 

    if (firsttime)
    {
	tempname = getenv( "EPHLOC" );
	if (tempname == NULL)
	{
	    printf(" could not get environment variable EPHLOC; \n");
	    printf(" it should tell directory containing translated JPL ephemeris data\n");
	    printf(" it is simplest if you set env-variable EPHLOC and re-run \n"); 
	    exit( 1 ); 
	}
	strncpy( ephloc, tempname, 127 ); 
	firsttime = 0; 
    }

    if ( fi != NULL  &&  strcmp( catalogFile, oldcatfile) == 0 )
    {
	if ( jd >= *tapeStart && jd <= *tapeEnd )
	{
	    return fi; /* don't need to do anything, nothing has changed */
	}
    }
    /* otherwise, start over: */ 

    if (fcat != NULL)
      fclose( fcat ); 
    fcat = fopen( catalogFile, "r" ); 
    if (fcat == NULL)
    {
	printf("in getDataFile," ); 
	printf(" could not open ephemeris catalog file %s \n", 
	       catalogFile ); 
	return (FILE *) NULL; 
    }

    strcpy( oldcatfile, catalogFile ); 

    /* this loop more or less assumes 
     * the files named in the cat-file are in temporal order 
     * that may not be the case . . . 
     */ 
    rewind( fcat ); 
    while ( 1 ) 
    {
	while( 1 )
	{		/* get next non-comment line from cat-file */ 
	    line = fgets( buf, 120, fcat ); 
	    if ( line == NULL)
	    {
		fclose( fcat );
		return (FILE *) NULL;
	    }
	    if ( !comment( line ) ) 
	      break; 
	}

	count = sscanf( line, "start: %lf end: %lf %s", 
		       &jdStart, &jdEnd, tempbuf ); 
	if (count < 3 )
	{
	    printf(" could not make sense of ephemeris catalog file %s;", 
		   catalogFile ); 
	    printf(" count = %d \n", count );
	    printf(" line = %s\n", line ); 
	    continue; 
	}
	if ( jd >= jdStart && jd<= jdEnd)
	{   /* found the right file, so open it and return */ 
	    /* here we tell calling routine the name of the file found: */ 
	    /* but first, concatenate it to ephloc to get a full path-name: */ 
	    strcpy( filename, ephloc );
	    strcat( filename, "/" );
	    strncat( filename, tempbuf, 126 ); 

	    if (fi != NULL) fclose( fi ); 
	    if ( strstr( filename, ".bin" ) != NULL )
	      fi = fopen( filename, "rb" );
	    else 
	      fi = fopen( filename, "r" );
	    oldfile = fi; 
	    if (fi == NULL)
	    {
		printf(" in getDataFile(), \n" ); 
		printf(" could not open file %s \n", filename );
		printf(" ephloc = %s \n", ephloc ); 
		fclose( fcat );
		return (FILE *) NULL; 
	    }
	    *tapeStart = jdStart;
	    *tapeEnd = jdEnd; 
	    return fi; 
	}
    }

    /* if you got to here, the catalog doesn't know a file for your date */ 
    fclose( fcat );
    return (FILE *) NULL; 

}


