


/* gmstgast.c 
 * 
 * sidereal time routines 
 *
 * copyright 1991 Andrew P. Porter
 */

#include <stdio.h>
#include <math.h>

#include "../include/ms.h" 
#include "../include/matrix.h"
#include "../include/hnav.h" 


double gmstx();		/* this routine is to be used only in this file
			 * it does not return GMST, only GMST at midnight
			 */


/* ----------------------------------------------------------------
 * GMST gives GMST as function of jd . . . 
 * GAST gives GAST as function of jd . . . 
 * 
 * input
 *	unix time, SI seconds 
 * output 
 *	GMST, in seconds of sidereal time 
 *	GAST, in seconds of sidereal time 
 * side effects 
 *	none 
 * source
 *	almanac, p. B-6, and Montenbruck, p. 37 top
 * 	correction from gmst to gast is from Montenbruck, p. 20-21
 * 	nutations can be had from your favorite source; see below 
 * ----------------------------------------------------------------
 */

double 
GMST( utime )
    double utime; 	/* unix time for which gmst is sought */
{
    double jd;	/* julian date of epoch for which GMST is sought */ 
    double T;		/* julian centuries from J 2000.0 */
    double gmst;	/* GMST to be returned */

    double gmsta;	/* gmst on midnight prior */  
    double gmstb; 	/* gmst on midnight after */  
    double Ta, Tb;	/* julian dates on midnight prior and after */ 
    double jda, jdb;	/* jd on midnights prior and after */ 

    jd = calendarUnixToJulian( utime ); 
    T  = ( jd - 2451545.0 ) / 36525.;
    /* (int) jd+0.5 is the jd on the noon of the day of interest */
    jda = (int) (jd + 0.5) - 0.5;           /* JD midnight before */ 
    jdb = jda + 1.;                         /* JD midnight after  */ 

    Ta = ( jda  - 2451545.0 ) / 36525.;     /* jul centuries midnight before */ 
    Tb = ( jdb  - 2451545.0 ) / 36525.;     /* jul centuries midnight after  */

    /* this is good only at midnight: */
    gmsta = gmstx( Ta );                    /* gmst midnight before */ 
    gmstb = gmstx( Tb );                    /* gmst midnight after  */ 

    /* in one day, midnight to midnight,
     * we go from gmsta to gmstb, through one revolution;
     * so interpolate between gmsta and (gmstb+86400.),
     * which is gmstb plus one rev;
     * (if you go directly from gmsta to gmstb, you get GMST
     * that is right only once a day, at midnight.)
     */
    gmst = gmsta * (jdb-jd)  + (gmstb + 86400.) * (jd-jda); 
    gmst = dnormalize( gmst, 0., 86400. );

    return gmst; 

}


/* use this routine only in this file: it is not in almanac.h */
/* it returns gmst  good ONLY  at midnight */
double
gmstx( T ) 
     double T; 		/* time in Julian centuries from J 2000.0 */ 
{
    double gmst;	/* temp storage */ 


    gmst = 24110.54841  + T * 8640184.812886 + T*T * 0.093104
		- T*T*T * 6.2e-6;

    gmst = dnormalize( gmst, 0., 86400. ); 

    return gmst; 
}


double 
GAST( utime )
    double utime; 	/* unix time for which gmst is sought */
{
    double jd;		/* julian day */ 
    double T;		/* julian centuries from J 2000.0 */  
    double gmst;	/* GMST */
    double gast; 	/* GAST to be returned */ 
    double dpsi;	/* nutation in longitude */
    double deps;	/* not used */
    double obliquity;	/* obliquity of the eccliptic */

    gmst = GMST( utime );

    obliquity = meanObliquity( utime ); 
    jd = calendarUnixToJulian( utime );
    T = ( jd - 2451545.0 ) / 36525.;

    nod_( &T, &dpsi, &deps ); 
    /* gnod( &T, &dpsi, &deps ); */

    dpsi /= 15.; 		/* arc-seconds to time-seconds */
    gast = gmst + dpsi*cos(obliquity); 	/* Montenbruck, p. 38  */

    gast = normalize( gast, 0., 86400. );
    return gast; 
      
}
 

