/* jplanets.c -- routines to get planet positions from jpl ephemeris */ 


#include <stdio.h>
#include <stdlib.h>
#include <ctype.h>
#include <math.h>
#include <string.h>


#include "../include/nrutil.h"
/* link with nrutil.o, in same directory */

#include "../include/ms.h" 
#include "../include/matrix.h"
#include "../include/hnav.h" 


extern double normalize();

/* ----------------------------------------------------------------
 * routines in this file:

    these routines progress from most raw (jplEphemeris) 
    to most user-convient (ecBody):


    ecBody();           planetary pos & vel in ecliptic coords
    jplBody();          planetary pos & vel in equatorial coords
                        both find data files per catalog file 

    jplCenter();        equatorial planetary pos & vel from a data record
                        that you have already gotten from a data file,
			relative to sun, earth, or barycenter 

    jplPosVel();        equatorial planetary pos & vel from a data record
                        that you have already gotten from a data file 
			relative to barycenter (the format of the ephemeris)
			in the units of your choice 

    jplEphemeris();     equatorial pos & vel in units of the data file 

    jplNutations();     nutations from the ephemeris 
                        (i. e., from JPL's numerical integration, not the USNO
			trigonometric series evaluated in nod.f/nod.c,
			or approximated in gnod().  )

 *
 * ----------------------------------------------------------------
 */ 




/* ----------------------------------------------------------------
 * ecBody calls jplBody, then transforms to ecliptic coords of 2451545.0
 * ----------------------------------------------------------------
 */

int ecBody( jd, catalogFile, body, center, units, position, velocity )
     double jd;		/* date of requested positions */ 
     char *catalogFile;	/* name of catalog file; 
			   tells where the ephemeris files are */ 
     int body;		/* index of body: 0-8 = planets, 9 = moon, 10 = sun */ 
     int center;	/*  flag, indicates desired center of coordinate system */ 
     int units;		/* MKS or AUDAY */ 
     Coord3d *position;	/* answer, position */
     Coord3d *velocity; /* answer, velocity */ 
{
    int i; 

    i = jplBody( jd, catalogFile, body, center, units, position, velocity );
    toEcliptic( 2451545.0, position );
    toEcliptic( 2451545.0, velocity );


    return i; 
}



/* ----------------------------------------------------------------
 * jplBody gets a body position, referred to the center of your choice
 * inputs 
 * 	jd, julian date 
 *	a catalog file name: tells where to get the ephemeris  
 *	body index
 *	a flag:  sun-centered or bary-centered 
 *	Coord3d for position 
 *	Coord3d for velocity
 * outputs 
 *	1 for success, 0 for failure 
 * side effects 
 *	sets position, someday will also set velocity
 * units
 *	AU, AU/day, or M, M/sec -- you choose 
 * coordinate system: see jplEphemeris()  
 * ----------------------------------------------------------------
 */

int 
jplBody( jd, catalogFile, body, center, units, position, velocity )
     double jd;		/* date of requested positions */ 
     char *catalogFile;	/* name of catalog file; 
			   tells where the ephemeris files are */ 
     int body;		/* index of body: 0-8 = planets, 9 = moon, 10 = sun */ 
     int center;	/*  flag, indicates desired center of coordinate system */ 
     int units;		/* MKS or AUDAY */ 
     Coord3d *position;	/* answer, position */
     Coord3d *velocity; /* answer, velocity */ 
{
    static TapeHeap *heap=NULL;	/* the ephemeris data heap */ 
    static TapeHeap  heapbuf; 
    int success;		/* whether called routines succeeded or not */



    /* ---------------------------------------------------------------- */ 
    /* check for valid body argument */
    if (   !( (body >= 0 && body <= 10) || body == EARTHINDEX )    )
    {
	printf(" you called jplBody with body = %d (bad!) \n", body );
	exit(0);
    }

    /* ---------------------------------------------------------------- */
    /* get the data-heap for the date desired */ 

    if ( needHeap( catalogFile, heap, jd )  ) 
    {
	heap = catToHeap( jd, catalogFile );
	if (heap == NULL) 
	  return 0; 
	memcpy( &heapbuf, heap, sizeof( TapeHeap ) ); 
	heap = &heapbuf; 
    }


    /* ---------------------------------------------------------------- */
    /* now interpret the data-heap */ 
    success = jplCenter( jd, heap, body, center, units, position, velocity );

    if (success != 1)
    {
	printf(" in jplBody, jplCenter bombed \n");
	printf(" jd = %12.3f, body = %d \n", jd, body ); 
	printf(" heap  %d, heap->jdStart  %12.3f, heap->jdEnd  %12.3f \n", 
	       heap, heap->jdStart, heap->jdEnd ); 
	printf(" catalogFile = %s \n", catalogFile ); 

	exit(0); 
    }
  
    return success; 
}


/* ----------------------------------------------------------------
 * jplCenter manages center-of-coord-system transformations and such 
 *	and calls routines to go and get positions for you 
 * inputs 
 * 	jd, julian date 
 *	a data-heap 
 *	body index
 *	a flag:  sun-centered or bary-centered 
 *	Coord3d for position 
 *	Coord3d for velocity
 * outputs 
 *	1 for success, 0 for failure 
 * side effects 
 *	sets position, someday will also set velocity
 * units
 *	AU, AU/day, or M, M/sec -- you choose 
 * coordinate system: see jplEphemeris()  
 * ----------------------------------------------------------------
 */

int 
jplCenter( jd, heap, body, center, units, position, velocity )
     double jd;		/* date of requested positions */ 
     TapeHeap *heap;	/* where the data is */
     int body;		/* index of body: 0-8 = planets, 9 = moon, 10 = sun */ 
     int center;	/*  flag, indicates desired center of coordinate system */ 
     int units;		/* MKS or AUDAY */ 
     Coord3d *position;	/* answer, position */
     Coord3d *velocity; /* answer, velocity */ 
{
    Coord3d centerPos;		/* position of the center body */
    Coord3d centerVel;  	/* velocity of the center body */ 
    static double emrat=81.300587; /* earth moon mass ratio; 
				     default is jpl's value */ 
    Coord3d moonpos;		/* moon position */
    Coord3d moonvel;		/* moon velocity */ 
    double correctionFactor; 
    Coord3d earthPos;		/* earth-moon system position */
    Coord3d earthVel;		/* earth-moon system velocity */ 
    Coord3d tempvec;		/* a temporary place to put a vector */ 
    int success; 		/* whether called routines succeeded */ 

    /* ---------------------------------------------------------------- */
    /*  interpret the data-heap */ 
    /* printf(" entering jplCenter,      heap %x \n", heap ); */ 

    if (body != EARTHINDEX)
    {
	success = jplPosVel( jd, heap, body, units, position, velocity ); 	
	if (success == 0) 
	{
	    printf(" in jplCenter 1; \n\
                   jplPosVel bombed, jd %12.3f, heap %x, body %d \n", 
		   jd, heap, body ); 
	    return 0; 
	}
    }

    /* correct from earth-moon barycenter to earth, if earth is sought  */ 
    if ((body == EARTHINDEX)    ||
	(body == MOONINDEX)     || 
	(center == EARTHCENTERED)   )
    {
	/* correct from earth-moon barycenter to earth itself */ 
        success = jplPosVel( jd, heap, EARTHMOONBARYCENTER, 
			    units, &earthPos, &earthVel ); 
	if (success == 0) 
	{
	    printf(" in jplCenter 2; \n\
                   jplPosVel bombed, jd %12.3f, heap %x, body %d \n", 
		   jd, heap, body ); 
	    printf(" heap->jdStart %12.2f, heap->jdEnd %12.2f \n",
		   heap->jdStart, heap->jdEnd ); 
	    return 0; 
	}
	success = jplPosVel( jd, heap, MOONINDEX, units, &moonpos, &moonvel );
	if (success == 0) 
	{
	    printf(" in jplCenter 3; \n\
                   jplPosVel bombed, jd %12.3f, heap %x, body %d \n", 
		   jd, heap, body ); 
	    return 0; 
	}
	correctionFactor = 1./( 1. + emrat ); 

	GeoScalarMultVector( correctionFactor, &moonpos, &tempvec ); 
	GeoSubtractVector( &earthPos, &tempvec, &earthPos );

	GeoScalarMultVector( correctionFactor, &moonvel, &tempvec );
	GeoSubtractVector( &earthVel, &tempvec, &earthVel );  
    }

    if (center == BARYCENTERED) 
    {
	if (body != EARTHINDEX && body != MOONINDEX)
	  return 1; 
	if (body == EARTHINDEX)
	{
	    GeoScalarMultVector( 1., &earthPos, position );
	    GeoScalarMultVector( 1., &earthVel, velocity );
	    return 1; 	/* all done */ 
	}
	if (body == MOONINDEX)
	{
	    GeoAddVector( position, &earthPos, position );
	    GeoAddVector( velocity, &earthVel, velocity ); 
	    return 1; 
	}
    }
    if (center == SUNCENTERED)	/* subtract to heliocentric coords */ 
    {
	/* sun in barycentric coords: */ 
	success = jplPosVel( jd, heap, SUNINDEX, units, &centerPos, &centerVel ); 
	if (success == 0) 
	{
	    printf(" in jplCenter 4; \n\
                   jplPosVel bombed, jd %12.3f, heap %x, body %d \n", 
		   jd, heap, body ); 
	    return 0; 
	}
	if (body == MOONINDEX)	/* moon w/r/t earth + earth w/r/t barycenter */ 
	{			/* == moon w/r/t barycenter: */
	    GeoAddVector( position, &earthPos, position ); 
	    GeoAddVector( velocity, &earthVel, velocity );
	}
	if (body == EARTHINDEX)
	{				/* earth w/r/t barycenter */ 
	    GeoScalarMultVector( 1., &earthPos, position );
	    GeoScalarMultVector( 1., &earthVel, velocity );
	}
	/* body w/r/t barycenter: */
	GeoSubtractVector( position, &centerPos, position );	
	/* minus sun w/r/t barycenter --> body w/r/t sun: */
	GeoSubtractVector( velocity, &centerVel, velocity );	
	return 1; 
    }

    if (center == EARTHCENTERED)	/* subtract to earth centered coords */ 
    {
	if (body == MOONINDEX) 
	  return 1; 	/* moon is already relative to earth */ 
	if (body == EARTHINDEX)
	{
	    GeoZeroVector( position );
	    GeoZeroVector( velocity );
	    return 1; 
	}	/* sun and planets: */ 
	GeoSubtractVector( position, &earthPos, position ); 
	GeoSubtractVector( velocity, &earthVel, velocity ); 
	return 1; 
    }

    printf(" unknown or uncoded center flag == %d \n", center );
    return 0; 

}



/* ----------------------------------------------------------------
 * jplPosVel calls jplEphemeris for you, and converts units 
 * inputs 
 *	see arg list 
 * outputs 
 *	1 for success, 0 for failure 
 * side effects 
 * 	sets position and velocity, in units of choice 
 * ----------------------------------------------------------------
 */

int 
jplPosVel( jd, heap, body, units, position, velocity )
     double jd;
     TapeHeap *heap;
     int body;
     int units;
     Coord3d *position;
     Coord3d *velocity;
{
    int success;

    /* body in barycentered coords: */
    success = jplEphemeris( jd, heap, body, position, velocity ); 	

    if (success == 0) return 0; 

    if (units == AUDAY)
    {
      /* from Km to AU */
      GeoScalarMultVector( 1000./AU, position, position ); 	
      /* Km/sec to AU/day */ 
      GeoScalarMultVector( 86.4e6/AU, velocity, velocity ); 	
    }
    if (units == MKS)
    {
      /* from Km to Meters */
      GeoScalarMultVector( 1000., position, position ); 	
      /* Km/sec to M/sec */ 
      GeoScalarMultVector( 1000., velocity, velocity ); 	
    }
      
    return 1;

}




/* ----------------------------------------------------------------
 * jplEphemeris gets the geometric position of a body in coords of J 2000.0 
 * inputs 
 * 	jd, julian date 
 *	a tape heap
 *	body index
 *	Coord3d for position 
 *	Coord3d for velocity
 * outputs 
 *	1 for success, 0 for failure 
 * side effects 
 *	sets position, someday will also set velocity
 * units
 *	KM, kilometers !!!! YOU must scale to AU or meters
 * reference system
 *	barycenter of the solar system 
 * 	equatorial coordinates, 
 *         close to mean equator and equinox of J 2000.0 (fk5) 
 * 	but not exactly so . . . . yrrrch
 * ----------------------------------------------------------------
 */

int 
jplEphemeris( jd, heap, body, position, velocity )
     double jd;		/* date of requested positions */ 
     TapeHeap *heap;	/* contains the ephemeris data */
     int body;		/* index of body: 0-8 = planets, 9 = moon, 10 = sun */ 
     Coord3d *position;	/* answer, position */
     Coord3d *velocity; /* answer, velocity */ 
{
    double f;		/* function from chebeval */
    double fdot;	/* its derivative */ 
    double t;		/* a parameter in the interval [0,1]: 
			   tells how far into the sub-interval we are */ 

    int subindex;	/* tells which sub-interval we are in */ 
    double  fsubindex;	/* double version of subindex */ 
    int ncoeffs;	/* number of coefficients in the Chebyshev series */ 
    double *startOfArray;	/* ptr to start of coeffs in the data array */ 
    double intervalSize;	/* length/duration of a data record */ 
    double subIntervalSize;	/* length of a sub-interval */ 
    int component;		/* 0-1-2, for x-y-z */ 
    static int firsttime=1;
    static FILE *fl; 

    if (firsttime)
    {
	fl = logfile();
	firsttime = 0;
    }

    if (jd < heap->jdStart) return 0; 
    if (jd > heap->jdEnd  ) return 0; 
    if ( body < 0   ||   body >= sevenSize ) return 0; 

    intervalSize = heap->groupFour.recordLength;
    subIntervalSize = intervalSize / (double) heap->groupSeven.nintervals[body];  
    fsubindex = ( jd - heap->jdStart) / subIntervalSize ;

    subindex = (int) fsubindex; 

     /* these work except at the end of the interval, a special case: */ 
    if (subindex >= heap->groupSeven.nintervals[body] - 1  )
    {
	subindex = heap->groupSeven.nintervals[body] - 1; 
    }
    t = fsubindex - subindex;

    if (t < 0.  ||  t > 1.)
    { 	
	printf("in jplEphemeris, something is wrong . . . t = %e \n", t );
	fprintf( fl, "in jplEphemeris, something is wrong . . . t = %e \n", t );
	fprintf( fl, "   jd = %22.20f, bodyindex = %d, heap->jdStart = %12.2f, ", 
		jd, body, heap->jdStart ); 
	fprintf( fl, "heap->jdEnd = %12.2f \n", heap->jdEnd ); 
	fprintf( fl, "   subindex = %d, fsubindex = %e \n", subindex, fsubindex ); 

	fprintf( fl, "  in jplEphemeris(), t = %f, body %d, sub-index %d \n", 
		t, body, subindex ); 
	for (component=0; component<3; ++component )
	{
	    fprintf( fl, " heap->index [%d] [%d] [%d]  =  %d \n", 
		    body, subindex, component,  
		    heap->index[body][subindex][component] ); 
	}
	return 0; 
    }

    ncoeffs = heap->groupSeven.ncoeffs[body]; 
      
    component = 0; 
    startOfArray = &(heap->data[ heap->index[body][subindex][component]  ]  ); 
    chebeval( t, startOfArray, ncoeffs, &f, &fdot ); 
    position->x = f;
    velocity->x = fdot;

    component = 1; 
    startOfArray = &(heap->data[ heap->index[body][subindex][component]  ]  ); 
    chebeval( t, startOfArray, ncoeffs, &f, &fdot ); 
    position->y = f;
    velocity->y = fdot;

    /* nutations have only two components ... others have three: */ 
    if (body != NUTATIONS)
      {
	component = 2; 
	startOfArray = &(heap->data[ heap->index[body][subindex][component]  ]  ); 
	chebeval( t, startOfArray, ncoeffs, &f, &fdot ); 
	position->z = f;
	velocity->z = fdot;
      }

    /* scale the velocity for the (sub)interval size: */ 
    /* the scale factor: 
       * 2. is for the shift from a chebyshev domain of [-1,1] to [0,1] ???? 
       * the subIntervalSize is in days, 
       *         so convert to seconds; output is in km/sec
       * (and we have to divide by the (sub) interval size 
       *         to get units of per-time)
       */
    GeoScalarMultVector( 2./(subIntervalSize*86400.), velocity, velocity );
    
    return 1; 

}



/* ----------------------------------------------------------------
 * jplNutations gets nutations from the ephemeris 
 * inputs
 *	julian date
 *	ptr to the data heap
 *	ptr to where to put d-psi
 *	ptr to where to put d-eps
 * outputs 
 *	1 for success, 0 for failure
 * side effects
 *	sets d-psi and d-eps for you
 * comments
 *	it might be better to use this than nod.f or gnod ... ?? 
 * UNITS
 *      arc-seconds, just like nod_() and gnod() 
 * ----------------------------------------------------------------
 */

int
jplNutations( jd, heap, dpsi, deps )
     double jd;		/* date for which nutations are sought */
     TapeHeap *heap;	/* the data-heap which contains them */
     double *dpsi;	/* where to put nutation in longitude */
     double *deps; 	/* where to put nutation in obliquity */
{
    double f;		/* function from chebeval */
    double fdot;	/* its derivative */ 
    double t;		/* e [0,1]: tells how far into the sub-interval we are */ 

    int subindex;	/* tells which sub-interval we are in */ 
    double fsubindex;	/* double version of subindex */ 
    int ncoeffs;	/* number of coefficients in the Chebyshev series */ 
    double *startOfArray;	/* ptr to start of coeffs in the data array */ 
    double intervalSize;	/* length/duration of a data record */ 
    double subIntervalSize;	/* length of a sub-interval */ 
    int component;		/* 0-1-2, for x-y-z */ 

    if (jd < heap->jdStart) return 0; 
    if (jd > heap->jdEnd  ) return 0; 

    intervalSize = heap->groupFour.recordLength;
    subIntervalSize = intervalSize / (double) heap->groupSeven.nintervals[NUTATIONS];  
    fsubindex = ( jd - heap->jdStart )  /  subIntervalSize ;

    subindex = (int) fsubindex; 
     /* this works except at the end of the interval, a special case: */ 
    if (subindex >= heap->groupSeven.nintervals[NUTATIONS] - 1  )
    {
	subindex = heap->groupSeven.nintervals[NUTATIONS] - 1; 
    }
    t = fsubindex - subindex;

    if (t < 0.  ||  t > 1.)
    { 	
	printf(" something is wrong . . . t = %e \n", t );
	return 0; 
    }

    if (t == 1.) printf(" t = %e \n", t ); 	/* remove this printf */ 
    /* NUTATIONS is the "body" number of nutations: */
    ncoeffs = heap->groupSeven.ncoeffs[NUTATIONS]; 	

    startOfArray = &(heap->data[ heap->index[NUTATIONS][subindex][0]  ]  ); 
    chebeval( t, startOfArray, ncoeffs, &f, &fdot ); 
    *dpsi = f * TODEGREES;  /* the ephemeris is in radians (good!) */
    *dpsi *= 3600.;         /* now convert to arc-seconds */ 

    /* we have not converted fdot, time deriv. of nutations, 
     * from radians to arc-seconds because it is not used;
     * if you decide to use it, don't forget to do the conversion ...
     */ 

    /* very very bad; nutations greater than 1 radian
     *  seems to be an outside index of implausible answers
     */ 
    if (f > 1.)
    {
      printf("\n in jplNutations, f (i. e., dpsi) = %e > 1. (very bad) \n", f ); 
      printf(" t = %e \n", t ); 
      printf(" jd %12.2f, jdstart %12.2f, jdend %12.2f \n", 
	     jd, heap->jdStart, heap->jdEnd ); 
      printf(" subindex = %d, fsubindex = %e \n", subindex, fsubindex );
    }

    startOfArray = &(heap->data[ heap->index[NUTATIONS][subindex][1]  ]  ); 
    chebeval( t, startOfArray, ncoeffs, &f, &fdot ); 
    *deps = f * TODEGREES;   /* radians to degrees */
    *deps *= 3600.;          /* degrees to arc-seconds */ 

    return 1; 

}
