/* matrix.c
 * 88/03/10
 * 
 * matrix and vector manipulation routines
 * routines with Geo- prefix were written originally by Walter Scott
 * routines with mat- prefix were written originally by Andrew Porter
 * modifications have been made by others in O Division 
 */


#include <stdio.h>
#include <math.h>


#include "../include/ms.h"
#include "../include/matrix.h"

#include "../include/hnav.h" 



/* 
 * ---------------------------------------------------------------- 
 * 
 * GeoZeroVector
 * 
 * store (0,0,0) in vector dd
 *
 * results:
 *	none;
 *
 * side effects:
 * 	see above.
 *
 * ---------------------------------------------------------------- 
 */ 
void
GeoZeroVector(dd)
     register Coord3d *dd;
{
  dd->x = 0.;
  dd->y = 0.;
  dd->z = 0.;
}  


/*
 * ----------------------------------------------------------------------------
 *
 * GeoUnitVector --
 *
 * Stores in *unit a unit vector in the direction of *c.
 * If *c is the null vector, *unit will be the null vector too.
 *
 * Results:
 *	FALSE if *c was NULL, TRUE otherwise.
 *
 * Side effects:
 *	See above.
 *
 * ----------------------------------------------------------------------------
 */

bool
GeoUnitVector(c, unit)
    register Coord3d *c, *unit;
{
    register double len = GeoVectorLength(c);

    if (len == 0.0)
    {
	*unit = *c;
	return FALSE;
    }

    unit->x = c->x / len;
    unit->y = c->y / len;
    unit->z = c->z / len;
    return TRUE;
}

/*
 * ----------------------------------------------------------------------------
 *
 * GeoNegateVector --
 *
 * Stores in 'areverse' the vector that's the negative of 'a'.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	See above.
 *
 * ----------------------------------------------------------------------------
 */

void
GeoNegateVector(a, areverse)
    register Coord3d *a, *areverse;
{
    areverse->x = - a->x;
    areverse->y = - a->y;
    areverse->z = - a->z;
}

/* 
 * ---------------------------------------------------------------- 
 * 
 * GeoAddVector --
 * 
 * compute the vector sum a + b and store it in c. 
 *
 * results:
 *	none;
 *
 * side effects:
 * 	see above.
 *
 * ---------------------------------------------------------------- 
 */ 
void
GeoAddVector(a,b,c)
     register Coord3d *a, *b, *c;
{
  c->x = a->x + b->x;
  c->y = a->y + b->y;
  c->z = a->z + b->z;
}


/*
 * ----------------------------------------------------------------------------
 *
 * GeoSubtractVector --
 *
 * Compute the vector difference a - b and store it in c.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	See above.
 *
 * ----------------------------------------------------------------------------
 */

void
GeoSubtractVector(a, b, c)
    register Coord3d *a, *b, *c;
{
    c->x = a->x - b->x;
    c->y = a->y - b->y;
    c->z = a->z - b->z;
}


/* 
 * ---------------------------------------------------------------- 
 * 
 * GeoScalarMultVector
 * 
 * compute the vector s*v and store it in c
 *
 * results:
 *	none;
 *
 * side effects:
 * 	see above.
 *
 * ---------------------------------------------------------------- 
 */ 

void
GeoScalarMultVector(s,v,c)
     double s;
     register Coord3d *v, *c;
{
  c->x = s*v->x;
  c->y = s*v->y;
  c->z = s*v->z;
}


/*
 * ----------------------------------------------------------------------------
 *
 * GeoDotProduct --
 *
 * Compute the dot product of the vectors a and b and return it.
 *
 * Results:
 *	See above.
 *
 * Side effects:
 *	None.
 *
 * ----------------------------------------------------------------------------
 */

double
GeoDotProduct(a, b)
    register Coord3d *a, *b;
{
    return a->x * b->x + a->y * b->y + a->z * b->z;
}

/*
 * ----------------------------------------------------------------------------
 *
 * GeoCrossProduct --
 *
 * Compute the cross product of the vectors a and b, leaving it in c.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	See above.
 *
 * ----------------------------------------------------------------------------
 */

void
GeoCrossProduct(a, b, c)
    register Coord3d *a, *b, *c;
{
    c->x = a->y * b->z - a->z * b->y;
    c->y = a->z * b->x - a->x * b->z;
    c->z = a->x * b->y - a->y * b->x;
}

/*
 * ----------------------------------------------------------------------------
 *
 * GeoVectorLength --
 *
 * Return the length of the vector given by *c.
 *
 * Results:
 *	Just told you.
 *
 * Side effects:
 *	None.
 *
 * ----------------------------------------------------------------------------
 */

double
GeoVectorLength(c)
    register Coord3d *c;
{
    return sqrt( c->x*c->x + c->y*c->y + c->z*c->z );
}

/*
 * ----------------------------------------------------------------------------
 *
 * GeoDistance --
 *
 * Return the distance between the two points c1, c2.
 *
 * Results:
 *	Just told you.
 *
 * Side effects:
 *	None.
 *
 * ----------------------------------------------------------------------------
 */

double
GeoDistance(c1, c2)
    register Coord3d *c1, *c2;
{
    register double xd, yd, zd;

    xd = c1->x - c2->x;
    yd = c1->y - c2->y;
    zd = c1->z - c2->z;
    return sqrt( xd*xd + yd*yd + zd*zd );
}


/* ----------------------------------------------------------------
 * moved here from .../ode/odeAdams.c 
 * ----------------------------------------------------------------
 */

void
GeoCopyVector(a,b)
     Coord3d *a;
     Coord3d *b;
{
    b->x = a->x;
    b->y = a->y;
    b->z = a->z;
}

double  
matAngleofVectors( a, b )
     Coord3d *a;
     Coord3d *b;
{
    double A, B, temp; 

    A = GeoVectorLength( a );
    B = GeoVectorLength( b );

    
    if ( A != 0.  &&  B != 0.)
    {
	temp = GeoDotProduct( a, b ) / (A*B);
	if (temp >  1.) temp = 1.;
	if (temp < -1.) temp = -1.; 
	return acos( temp ); 
    }
    else 
      return 0.;
}




/* ----------------------------------------------------------------
 * getfile does what you do so often 
 * gets a file hook and complains and quits if it fails 
 * ----------------------------------------------------------------
 */

FILE *
getfile( filename, mode )
     char *filename;
     char *mode;
{
    static FILE *fo; 	/* this is what the caller wants */ 

    fo = fopen( filename, mode );
    if (fo == NULL)
    {
	printf(" could not open file %s \n", filename ); 
	exit(0);
    }
    
    return fo; 
}




/* ----------------------------------------------------------------
 * normalize --
 * #### #### #### #### remove one of normalize(), dnormalize() ... !!!! 
 * returns normalized version of double first arg to fall between second and third args
 * inputs
 *	x, value to be normalized
 *	low, lower bound of normal range
 *	high, upper bound of normal range
 * outputs 
 *	normalized version of x
 * side effects
 *	none
 * ----------------------------------------------------------------
 */

double
normalize(x,low,high)
     double x;		/* number to be normalized */
     double low;		/* low end of normal range */
     double high;	/* high end of normal range */
{
  double difference;	/* the "period" by which we normalize */
  double xx;		/* here we construct the normalized number */ 
  int longjump;		/* integer number of periods away */
  double answer;		/* the final answer */

  difference = high - low; 
  xx = x;
  longjump = (xx - low) / difference;
  xx -= difference * longjump;

  while (xx <  low)      xx += difference;
  while (xx >= high)     xx -= difference;

  answer = xx; 
  return answer;
}

double 
dnormalize( x, low, high )
     double x;		/* number to be normalized */
     double low;		/* low end of normal range */
     double high;	/* high end of normal range */
{
  double difference;	/* the "period" by which we normalize */
  double xx;		/* here we construct the normalized number */ 
  int longjump;		/* integer number of periods away */
  double answer;		/* the final answer */

  difference = high - low; 
  xx = x;
  longjump = (xx - low) / difference;
  xx -= difference * longjump;

  while (xx <  low)      xx += difference;
  while (xx >= high)     xx -= difference;

  answer = xx; 
  return answer;
}



/* 
 * ----------------------------------------------------------------
 * matInvert reads m and sets x, both 3x3 matrices
 * 	it calls for determinant (sets x to m and quits if zero)
 * 	sets each element by hand with loving care
 * inputs
 *	3x3 array to be inverted
 *	3x3 array to hold the inverse	
 * outputs
 * 	returns 1 for success, 0 for failure
 * side effects
 *	sets the second argument to the inverse matrix of the first argument
 */

int 
matInvert(m,x)
     double m[3][3];
     double x[3][3];
{
  void  matDump(), matDiff(), matMult(); 
  double d; 
  double toosmall; 	/* cutoff below which determinant is considered zero */
  double size;		/* an l2 norm of the matrix */
  double p[3][3];	/* to hold product of matrix and its inverse */
  double df[3][3];	/* matrix temp storage */
  double id[3][3];	/* identity matrix */
  double div;		/* reciprocal of the determinant */

	/* set the identity matrix . . . */
  matUnitMatrix(id);

  size = matSize(m);  				/* get the determinant: */
  toosmall = 1.e-6*sqrt(size);
  d = matDeterminant(m);
  if (ABS(d) <= toosmall) 
  {
      matCopy(m,x);  
      printf("in matinvert; determinant = %e \n",d);
      return 0;  
  }

  div = 1. / d;  				/* set the inverse matrix elements */
  x[0][0] =  ( m[1][1]*m[2][2] - m[2][1]*m[1][2] ) * div;
  x[0][1] = -( m[1][0]*m[2][2] - m[2][0]*m[1][2] ) * div;
  x[0][2] =  ( m[1][0]*m[2][1] - m[2][0]*m[1][1] ) * div;

  x[1][0] = -( m[0][1]*m[2][2] - m[2][1]*m[0][2] ) * div;
  x[1][1] =  ( m[0][0]*m[2][2] - m[2][0]*m[0][2] ) * div;
  x[1][2] = -( m[0][0]*m[2][1] - m[2][0]*m[0][1] ) * div;

  x[2][0] =  ( m[0][1]*m[1][2] - m[1][1]*m[0][2] ) * div;
  x[2][1] = -( m[0][0]*m[1][2] - m[1][0]*m[0][2] ) * div;
  x[2][2] =  ( m[0][0]*m[1][1] - m[1][0]*m[0][1] ) * div;

  matMult(m,x,p);
  matDiff(p,id,df);
  d = matSize(df);
  if (d > 1.e-7) {
    printf("\ninvert seems to have failed\n");
    matDump(p,"\nproduct of matrix and inverse");
    matDump(m,"\nmatrix to be inverted");
    matDump(x,"\ncandidate inverse matrix"); 
    return 0;
  }
  return 1;
}


/* ---------------------------------------------------------------- 
 *
 * reduce(x)
 * reduces angles: returns a value between PI and -PI
 * obsolete: use normalize instead  
 * ---------------------------------------------------------------- 
 */

double reduce(y)
double y;
{
  double tpi,x;
  x = y;
  tpi = 2.*PI;
  while (x <  -PI ) x += tpi;
  while (x >=  PI ) x -= tpi;
  return x;
}


/* ----------------------------------------------------------------
 * matUnitMatrix 
 * 	place a unit matrix in the argument
 * inputs
 *	3x3 array to hold result 
 * outputs
 *	none
 * side effects
 *	sets unit matrix in argument
 * ----------------------------------------------------------------
 */

void
matUnitMatrix(id)
     matrix3d id;
{

  id[1][0] = id[0][1] = id[1][2] = 0.;	
  id[2][1] = id[2][0] = id[0][2] = 0.;
  id[0][0] = id[1][1] = id[2][2] = 1.;
}



/* ----------------------------------------------------------------
 * matScalarMultMat: scalar multiplies matrix
 * inputs
 * 	scalar
 *	matrix in
 * 	matrix out
 * ouputs
 *	none
 * side effects
 *	matrix out = scalar X matrix in
 * ----------------------------------------------------------------
 */

void
matScalarMultMat (scalar, in, out)
     double scalar;		/* scalar multiplier */
     matrix3d in;		/* matrix in */
     matrix3d out;		/* matrix out */
{
    int i, j;	/* indices */

    for (i=0; i<3; ++i)
      for (j=0; j<3; ++j)
	out[i][j] = scalar * in[i][j];

}





/* ----------------------------------------------------------------
 * matSize returns sum of squares of the 9 elements
 * inputs
 *	3x3 array
 * outputs
 *	euclidean norm of the matrix 
 * side effects
 *	none
 * ----------------------------------------------------------------
 */


double 
matSize(m)
     double m[3][3];
{
  double size;
  size  = m[0][0]*m[0][0] + m[0][1]*m[0][1] + m[0][2]*m[0][2];
  size += m[1][0]*m[1][0] + m[1][1]*m[1][1] + m[1][2]*m[1][2];
  size += m[2][0]*m[2][0] + m[2][1]*m[2][1] + m[2][2]*m[2][2];
  return size;
}


/* ----------------------------------------------------------------
 * matDeterminant returns determinant of 3x3 matrix
 * inputs
 *	3x3 array
 * outputs
 *	determinant
 * side effects
 *	none
 * ----------------------------------------------------------------
 */

double 
matDeterminant(m)
     double m[3][3];
{
  double d;

  d =  m[0][0] * m[1][1] * m[2][2];
  d += m[2][0] * m[0][1] * m[1][2];
  d += m[1][0] * m[2][1] * m[0][2];

  d -= m[2][0] * m[1][1] * m[0][2];
  d -= m[0][0] * m[2][1] * m[1][2];
  d -= m[1][0] * m[0][1] * m[2][2];

  return d;
}


/* ----------------------------------------------------------------
 * matCopy copies input to output argument
 * inputs
 *	two 3x3 arrays
 * ouputs	
 *	none
 * side effects
 *	copies first 3x3 array to second 3x3 array
 * ----------------------------------------------------------------
 */


void 
matCopy(m,x)
     double m[3][3];
     double x[3][3];
{
    x[0][0] = m[0][0];
    x[0][1] = m[0][1];
    x[0][2] = m[0][2];

    x[1][0] = m[1][0];
    x[1][1] = m[1][1];
    x[1][2] = m[1][2];

    x[2][0] = m[2][0];
    x[2][1] = m[2][1];
    x[2][2] = m[2][2];
}



/* ----------------------------------------------------------------
 * matDiff creates the difference of two matrices 
 * inputs
 *	three 3x3 arrays
 * ouputs	
 *	none
 * side effects
 *	puts <first - second> in <third> matrix
 * ----------------------------------------------------------------
 */



void
matDiff(m,x,d)
     double m[3][3];
     double x[3][3];
     double d[3][3];
{
  int i,j;
  for (i=0; i<3; ++i) for (j=0; j<3; ++j) d[i][j] = m[i][j] - x[i][j];
}


/* ----------------------------------------------------------------
 * matMult multiplies two matrices 
 * inputs
 *	three 3x3 arrays
 * ouputs	
 *	none
 * side effects
 *	puts first X second in third matrix
 * note -- product can be either of first or second matrices 
 * ----------------------------------------------------------------
 */


void
matMult( m, x, prod )
     double m[3][3];
     double x[3][3];
     double prod[3][3];
{
  int i,j,k;	/* indices */
  double p[3][3]; 

  for (i=0; i<3; ++i)   
    for (j=0; j<3; ++j) 
      p[i][j] = 0.;

  for (i=0; i<3; ++i)
    for (j=0; j<3; ++j)
      for (k=0; k<3; ++k)
	p[i][j] += m[i][k]*x[k][j];

  for (i=0; i<3; ++i)   
    for (j=0; j<3; ++j) 
      prod[i][j] = p[i][j]; 


}




/* ----------------------------------------------------------------
 * matVMult, matMatVV
 * 	these routines differ only in order of calling arguments
 * 	and in the way the arguments are declared ( [] or * )	
 * pre-multiply a vector by a matrix
 *	the difference between pre- and post- multiply 
 *	amounts to transposing the matrix;
 *	these routines effectively transpose the matrix,
 *	compared to GeoTransCoord
 * inputs
 *	a source vector and a matrix, and a destination vector
 * NOTE: 
 * 	form of vector and matrix inputs is DIFFERENT for the two routines here
 * outputs
 *	none
 * side effects 
 *	sets destination vector
 * NOTE 
 *	you CAN have output vector == input vector
 * ----------------------------------------------------------------
 */

void
matVMult(invec,mat,outvec)
     double invec[3];
     double mat[3][3];
     double outvec[3];
{
    Coord3d v;	/* temp storage: this means you can call this routine with outvec == invec */

      v.x       =   invec[0]*mat[0][0] 
		  + invec[1]*mat[0][1]
		  + invec[2]*mat[0][2];  

      v.y       =   invec[0]*mat[1][0] 
		  + invec[1]*mat[1][1]
		  + invec[2]*mat[1][2];  

      v.z       =   invec[0]*mat[2][0] 
		  + invec[1]*mat[2][1]
		  + invec[2]*mat[2][2];  

      outvec[0] = v.x;
      outvec[1] = v.y;
      outvec[2] = v.z;

}


/* ----------------------------------------------------------------
 *  note that invec and outvec can be the same vector,
 * because the result is stored in the temporary <vec>;
 * don't speed up this routine by eliminating <vec>,
 * because some clients call it with invec = outvec 
 * ----------------------------------------------------------------
 */ 

void
matMatVV( mat, invec, outvec )
     matrix3d mat;
     Coord3d  *invec;
     Coord3d  *outvec;
{
    Coord3d vec;

      vec.x =       invec->x * mat[0][0] 
		  + invec->y * mat[0][1]
		  + invec->z * mat[0][2];  

      vec.y =       invec->x * mat[1][0] 
		  + invec->y * mat[1][1]
		  + invec->z * mat[1][2];  

      vec.z =       invec->x * mat[2][0] 
		  + invec->y * mat[2][1]
		  + invec->z * mat[2][2];  

    outvec->x = vec.x;
    outvec->y = vec.y;
    outvec->z = vec.z;

}


/*
 * ----------------------------------------------------------------
 * matERot, matEulerToMat
 * 	generate rotation matrix in mat[][]
 * 	from euler angles supplied as arguments 
 *	euler angles are interpreted as from world to rotated frame
 * 	the resulting matrix is from world to rotated frame, 
 *	IF you multiply with matVMult ! 
 *
 * inputs: for matERot
 *	3 angles
 *	place to put mat
 * inputs: for matEMat
 *	EulerRot *er
 *	place to put mat
 * outputs
 *	none
 * side effects
 *	sets mat[][]
 *
 * ----------------------------------------------------------------
 */

void 
matEulerToMat(er,mat)
     EulerRot *er;
     matrix3d mat;
{ matERot(er->er_1,er->er_2,er->er_3,mat); }


void
matERot( phi_rot, theta_rot, psi_rot, mat)
double phi_rot, theta_rot, psi_rot;	/* euler angles */
double mat[3][3];			/* the matrix to be created from them */
{

    double cpr,spr,ctr,str,cpsr,spsr;	/* sines, cosines */

    cpr = cos(phi_rot);
    spr = sin(phi_rot);   
    ctr = cos(theta_rot);
    str = sin(theta_rot); 
    cpsr = cos(psi_rot);
    spsr = sin(psi_rot);

    mat[0][0] = cpr*cpsr - spr*ctr*spsr;     /* rotation matrix; phi, about initial z; */
    mat[0][1] = -cpr*spsr - spr*ctr*cpsr;    /* and theta, about next x. */ 
    mat[0][2] = spr*str;		     /* psi about last z	 */
    mat[1][0] = spr*cpsr + cpr*ctr*spsr;     /* see Goldstein, p. 109, eq. (4-47), */
    mat[1][1] = -spr*spsr + cpr*ctr*cpsr;
    mat[1][2] = -cpr*str;
    mat[2][0] = str*spsr;
    mat[2][1] = str*cpsr;
    mat[2][2] = ctr;
  
}


/* ----------------------------------------------------------------
 * matMatToEuler puts euler angles constructed from mat in *er
 * 	see comments at matERot() above
 * inputs
 *	matrix
 * outputs
 *	none
 * side effects
 *	puts euler angles in *er	
 * ----------------------------------------------------------------
 */

void
matMatToEuler(m,er)
     matrix3d m;
     EulerRot *er;
{
  er->er_2 = acos(m[2][2]);
  if (er->er_2 == 0.) 
    {
      er->er_3 = 0.;
      er->er_1 = atan2( m[1][0], m[0][0]);	/* see Goldstein, p. 109, eqn. (4-47) */
    }
  else
    {
      er->er_1 = atan2( m[0][2], - m[1][2]);
      er->er_3 = atan2( m[2][0],   m[2][1]);
    }
  er->er_1 = normalize( er->er_1, 0., PI2 ); 
  er->er_3 = normalize( er->er_3, 0., PI2 ); 
}



/* ----------------------------------------------------------------
 * matTranspose transposes matrix
 * 	creates transpose of first arg in second arg
 * inputs
 *	two 3x3 arrays 
 * outputs
 *	none
 * side effects
 *	creates transpose of first arg in second arg
 */

void 
matTranspose(m,y)
     double m[3][3];
     double y[3][3];
{
    double x[3][3];

    x[0][0] = m[0][0];
    x[1][1] = m[1][1];
    x[2][2] = m[2][2];

    x[0][1] = m[1][0];
    x[0][2] = m[2][0];
    
    x[1][0] = m[0][1];
    x[1][2] = m[2][1];

    x[2][0] = m[0][2];
    x[2][1] = m[1][2];


    /* now transfer the transposed matrix to destination */ 
    y[0][0] = x[0][0];
    y[1][1] = x[1][1];
    y[2][2] = x[2][2];

    y[0][1] = x[1][0];
    y[0][2] = x[2][0];

    y[1][0] = x[0][1];
    y[1][2] = x[2][1];

    y[2][0] = x[0][2];
    y[2][1] = m[1][2];



}

void 
matMatTranspose(m,y)
     matrix3d m;
     matrix3d y;
{
     matrix3d x;

     x[0][0] = m[0][0];
     x[1][1] = m[1][1];
     x[2][2] = m[2][2];

     x[0][1] = m[1][0];
     x[0][2] = m[2][0];

     x[1][0] = m[0][1];
     x[1][2] = m[2][1];

     x[2][0] = m[0][2];
     x[2][1] = m[1][2];

     /* now transfer to destination: */ 
     y[0][0] = x[0][0];
     y[1][1] = x[1][1];
     y[2][2] = x[2][2];

     y[0][1] = x[1][0];
     y[0][2] = x[2][0];

     y[1][0] = x[0][1];
     y[1][2] = x[2][1];

     y[2][0] = x[0][2];
     y[2][1] = x[1][2];


}




/*
 * ----------------------------------------------------------------------------
 *
 * matPolarToRect --
 *
 * Convert polar to rectangular coordinates.
 *	NOTE       EAST LONGITUDE IS POSITIVE !!!!
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	Fills in *c with the rectangular representation of the polar
 *	coordinates in *pc.
 *
 * ----------------------------------------------------------------------------
 */

void
matPolarToRect(pc, c)
    register PCoord *pc;
    register Coord3d *c;
{
    register double coslat;

    coslat = cos(pc->pc_lat);
    c->x =   pc->pc_r * coslat * cos(pc->pc_long);
    c->y =   pc->pc_r * coslat * sin(pc->pc_long);
    c->z =   pc->pc_r * sin(pc->pc_lat);

}

/*
 * ----------------------------------------------------------------------------
 *
 * matRectToPolar --
 *	NOTE       EAST LONGITUDE IS POSITIVE !!!!
 *
 * Convert rectangular to polar coordinates.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	Fills in *pc with the polar representation of the rectangular
 *	coordinates in *c.
 *
 * ----------------------------------------------------------------------------
 */

void
matRectToPolar(c, pc)
    register Coord3d *c;
    register PCoord *pc;
{
    register double x2, y2;

    x2 = c->x * c->x;
    y2 = c->y * c->y;
    pc->pc_r = sqrt(x2 + y2 + c->z * c->z);
    if (pc->pc_r == 0.0)
    {
	pc->pc_long = pc->pc_lat = 0.0;
	return;
    }

    if (x2 == 0.0 && y2 == 0.0)
    {
	pc->pc_long = 0.0;
	if (c->z > 0.0) pc->pc_lat = PI / 2.0;
	else pc->pc_lat = -PI / 2.0;
	return;
    }

    pc->pc_lat = atan2(c->z, sqrt(x2 + y2));
    pc->pc_long = atan2( c->y, c->x);
    
}


/* ----------------------------------------------------------------
 * matAccumulateVector
 *	accumulates scalar*vector in sum
 * inputs
 *	scalar, 
 *	vector, 
 *	place to accumulate vector sum
 * outputs
 *	none
 * side effects
 *	puts the vector sum in *sum:
 * 	(vector) sum += scalar * (vector) vector
 * ----------------------------------------------------------------
 */ 

void
matAccumulateVector( scalar, vector, sum)
     double 	scalar;
     Coord3d 	*vector;
     Coord3d	*sum;
{
    Coord3d	temp;
    GeoScalarMultVector( scalar, vector, &temp);
    GeoAddVector( &temp, sum, sum);
}




/* ----------------------------------------------------------------
 * matPolarUnitVectors
 *	makes polar unit vectors, given polar position
 * inputs
 *	cartesian position
 *	ptr to array of 3 Coord3d's, in which to put unit vectors
 * outputs
 *	none
 * side effects
 *	puts 
 *		r_unit in uv[0],	direction of increasing r
 *		phi_unit in uv[2],	direction of increasing latitude
 *		lambda_unit in un[1];	direction of increasing longitude
 * note the order:
 *	in this order, they are in right-handed sequence;
 *	that is, uv[2] = uv[0] cross uv[1]
 * ----------------------------------------------------------------
 */ 

void
matPolarUnitVectors( rpos, uv)
     PCoord	*rpos;
     Coord3d	uv[3];
{
    Coord3d	z_unit;
    Coord3d	pos;
    double	uz;	/* sin(latitude) */

    matPolarToRect( rpos, &pos);

    /* the r unit-vector is easy: */
    GeoUnitVector( &pos, &(uv[0]) );
    
    z_unit.x = 0.;
    z_unit.y = 0.;
    z_unit.z = 1.;

    /* the lambda unit-vector is tricky at the poles: */
    if (uz >= 0.9999 || uz <= - 0.9999) 
    {
	uv[1].x = - sin(rpos->pc_long);
	uv[1].y =   cos(rpos->pc_long);
	uv[1].z = 0.;
    }
    else
    {
	GeoCrossProduct( &z_unit,  &(uv[0]), &(uv[1]) );
	GeoUnitVector(   &(uv[1]), &(uv[1]) );
    }

    /* the phi unit-vector follows from the other two: */
    GeoCrossProduct( &(uv[0]), &(uv[1]), &(uv[2]) );

}




/* ----------------------------------------------------------------
 * odeMatCheck checks selected routines in matrix.c
 * ----------------------------------------------------------------
 */

void
odeMatCheck()
{
    Coord3d z;
    PCoord  r;
    Coord3d uv[3];

    z.x = 1.;
    z.y = 0.;
    z.z = 0.;
    matRectToPolar( &z, &r);

    matPolarUnitVectors( &r, uv);
    dumpVec( &z,       "position");
    dumpVec( &(uv[0]), "r unit");
    dumpVec( &(uv[1]), "lambda unit" );
    dumpVec( &(uv[2]), "phi unit" );
    
    printf(" \n");
    z.x = 0.;
    z.y = 0.;
    z.z = 1.;
    matRectToPolar( &z, &r);

    matPolarUnitVectors( &r, uv);
    dumpVec( &z,       "position");
    dumpVec( &(uv[0]), "r unit");
    dumpVec( &(uv[1]), "lambda unit" );
    dumpVec( &(uv[2]), "phi unit" );
    
    printf(" \n");
    z.x = 0.;
    z.y = 1.;
    z.z = 0.;
    matRectToPolar( &z, &r);

    matPolarUnitVectors( &r, uv);
    dumpVec( &z,       "position");
    dumpVec( &(uv[0]), "r unit");
    dumpVec( &(uv[1]), "lambda unit" );
    dumpVec( &(uv[2]), "phi unit" );
    
    printf(" \n");

}


/* ----------------------------------------------------------------
 * ZXtoMatrix turns a Z-axis and an X-axis into a transform matrix
 *	i. e., from reference coords to coords with the given Z and X axes 
 *	it does NOT NOT NOT check that the input axes are perpendicular 
 * inputs
 *	Z axis
 *	X axis
 * 	matrix3d, in which to put the answer 
 * outputs
 *	none
 * side effects
 *	puts the transform matrix where you tell it to
 * calling locations (as of 94/06/07): 
 *      ../almanac/celcoord.c, planette.c
 * ----------------------------------------------------------------
 */

void
ZXtoMatrix( z, x, mat )
     Coord3d	*z;
     Coord3d	*x;
     matrix3d	mat;
{
    Coord3d	yy;
    Coord3d	xx;
    Coord3d	zz;
    Coord3d	bad;

    GeoUnitVector( z, &zz );
    GeoUnitVector( x, &xx );

    GeoScalarMultVector( GeoDotProduct( &xx, &zz ), &zz, &bad );
    GeoSubtractVector( &xx, &bad, &xx );
    GeoUnitVector( &xx, &xx );

    GeoCrossProduct( &zz, &xx, &yy );

    mat[0][0] = xx.x;
    mat[0][1] = xx.y;
    mat[0][2] = xx.z;
    mat[1][0] = yy.x;
    mat[1][1] = yy.y;
    mat[1][2] = yy.z;
    mat[2][0] = zz.x;
    mat[2][1] = zz.y;
    mat[2][2] = zz.z;

}

