
/* navAncilla.c
 * 
 * navigation software, support routines
 * 	low level calculations 
 *
 * copyright 1988-1994 Andrew P. Porter
 * 
 * much of this is of use only to earth-orbit satellite calculations,
 * but it has traditionally been in this file
 * and is of an exceedingly elementary nature
 */

#include <stdio.h>
#include <math.h>

#include "../include/ms.h"
#include "../include/matrix.h" 
#include "../include/hnav.h" 


Orbit *Xelements();
Orbit *Yelements();

#define TWO_32 4294967296
double dftime();



/* ----------------------------------------------------------------
 * procedure navGeocentLat
 * 	convert geodetic latitude to geocentric latitude 
 * 	geodetic latitude is latitude defined by local vertical
 * 	geocentric latitude is defined as angle at earth center rising from the equator
 * input:
 *	a geodetic latitude;
 * results:
 *	returns a geocentric latitude;
 * side effects:
 * 	none.
 * source
 *	Allen, Astrophysical Quantities, p. 114
 * misc comments
 * 	See diagrams, BMW, pp. 95-96, or see GBG, p. 563
 *	but BMW does not solve this problem: his 'beta' is not our answer
 * author: A. Porter
 * ----------------------------------------------------------------
 */
double 
navGeocentLat(geodetLat)
     double geodetLat;
{

    if (geodetLat >=  PIhalf) return  PIhalf;
    if (geodetLat <= -PIhalf) return -PIhalf;

    return geodetLat - (695.*TORADIANS/3600.) * sin(2.*geodetLat)
      + (1.2*TORADIANS/3600.) * sin(4.*geodetLat); 
}



/* ----------------------------------------------------------------
 * procedure navGeodetLat
 * 	convert geocentric latitude to geodetic latitude 
 * 	geodetic latitude is latitude defined by local vertical
 * 	geocentric latitude is defined as angle at earth center rising from the equator
 * input:
 *	a geocentric latitude;
 * results:
 *	returns a geodetic latitude;
 * side effects:
 * 	none.
 * sources
 *	BMW, p. 95-97 is somewhat helpful for this problem
 *	beta, in the fig. of p. 96, is an intermediate quantity
 *	eqRad * cos(beta) = true-radius * cos(geocentLat)
 *	tan(geodetLat) = (eqRad / polRad) tan(beta)	
 *	(eq. (2.8-3), p. 97, plus defn. of e, top of p. 96 and fig. on p. 95.)
 * see comments on sources at navGeocentLat
 * author: A. Porter
 * ----------------------------------------------------------------
 */
double 
navGeodetLat(geocentLat)
     double geocentLat;
{
    double radius;	/* true radius of the earth at this point */ 
    double beta;		/* the angle beta in fig. 2.8-2, p. 96 of BMW 
			   (beta is not our answer) */

    if (geocentLat >=  PIhalf) return  PIhalf;
    if (geocentLat <= -PIhalf) return -PIhalf;

    radius = navEarthRadius( geocentLat); 
    beta = acos( (radius / eqRad) * cos( geocentLat) );
    if (geocentLat < 0.) beta = - beta;
    return atan( (eqRad/polRad) * tan(beta ) );

}


/* ----------------------------------------------------------------
 * navCheckLat should verify that the previous two subroutines 
 *	are (approximately) true inverses of one another
 *	the fourth column in the printout should be approximately zero
 * input
 *	none
 * outputs
 *	none
 * side effects
 * 	only prints
 * results
 *	 when last run, the above two routines are inverses to within 15 microradians
 * author: A. Porter
 * ----------------------------------------------------------------
 */
  
void
navCheckLat()
{
    double detlat;	/* geodetic latitude */
    double centlat;	/* geocentric latitude */
    double inverse;	
    double difference;

    printf("geodet_lat, geocent_lat, geodet_lat(geocent lat), difference \n");
    for (detlat = -PIhalf; detlat <= PIhalf+0.01; detlat += 0.1*PI)
    {
	centlat = navGeocentLat( detlat);
	inverse = navGeodetLat( centlat); 
	difference = detlat - inverse;
	printf(" % 11f % 11f \n % 11f % 21f \n\n", 
	       detlat, centlat, inverse, difference );

    }

    printf("geocent_lat, geodet_lat, geocent_lat(geodet lat), difference \n"); 
    for (centlat = -PIhalf; centlat <= PIhalf+0.01; centlat += 0.1*PI)
    {
	detlat = navGeodetLat( centlat);
	inverse = navGeocentLat( detlat); 
	difference = centlat - inverse;
	printf(" % 11f % 11f \n % 11f % 21f \n\n", 
	       centlat, detlat, inverse, difference );
    }

}


/* ----------------------------------------------------------------
 * procedure navGeocentAlt 
 * inputs:
 *	a geodetic altitude, that is, altitude above surface, in meters
 * 	geocentric latitude, for getting local earth radius 
 * results:
 *	returns a geocentric altitude, that is, altitude from earth center, 
 *	in meters (about 6,621,000 meters);
 * side effects:
 * 	none.
 * author: A. Porter
 * ----------------------------------------------------------------
 */
double 
navGeocentAlt(geodetAlt,geocentLat)
     double geodetAlt;
     double geocentLat;
{
  double x;
  x = geodetAlt + navEarthRadius(geocentLat);
  return x;
}


/* ----------------------------------------------------------------
 * procedure navGeodetAlt 
 * inputs:
 *	a geocentric altitude, that is, altitude from earth center
 * 	geocentric latitude, for getting local earth radius 
 * results:
 *	returns a geodetic altitude, altitude above earth surface
 *	in meters (about 300,000 meters);
 * side effects:
 * 	none.
 * author: A. Porter
 * ----------------------------------------------------------------
 */
double  
navGeodetAlt(geocentAlt,geocentLat)
     double geocentAlt;
     double geocentLat;
{
  double x;
  x = geocentAlt - navEarthRadius(geocentLat);
  return x;
}

/* ----------------------------------------------------------------
 * procedure navGeocentLong (old name) navCelestialLong (clearer name) 
 * input:
 *	a geodetic longitude, that is, longitude above earth surface,
 * 		longitude east of Greenwich; 
 * 	time, UT1, seconds since 0h January 1, 1989
 * results:
 *	returns a geocentric longitude, 
 *	that is, celestial longitude, or right ascension;
 * side effects:
 * 	none.
 * see BMW, p. 99-100
 *	their lamda-e = geodetic longitude 
 *	their theta = geocentric longitude
 *	their theta-g = longitude of greenwich
 * 	theta = theta-g + lamba-e
 * author: A. Porter
 * ----------------------------------------------------------------
 */

double 
navCelestialLong(geodetLong,time)
     double geodetLong;
     double time;	/* UT1, seconds since 0h January 1, 1989 */
{
  double x;
  x = GAST( time ) + geodetLong;
  x = normalize( x, 0., PI2);
  return x;
}

/* old name: */
double 
navGeocentLong(geodetLong,time)
     double geodetLong;
     double time;	
{
    return navCelestialLong(geodetLong, time); 
}



/* ----------------------------------------------------------------
 * procedure navGeodetLong
 * input:
 *	a geocentric longitude, that is, celestial longitude or right ascension;
 * 	Greenwich Mean Time;
 * results:
 *	returns a geodetic longitude, that is, longitude east of Greenwich; 
 * side effects:
 * 	none.
 * author: A. Porter
 * ----------------------------------------------------------------
 */
double 
navGeodetLong(geocentLong,time)
     double geocentLong;
     double time; 	/* UT1, time since 0h January 1, 1989 */
{
  double x; 
  x = geocentLong - GAST( time ); 
  x = normalize( x, 0., PI2);
  return x;
}


/* ----------------------------------------------------------------
 * navEarthRadius
 * input:
 *	geoCENTRIC (NB!) latitude;
 * result: 
 *	radius of the earth at this geocentric latitude;
 * side effects:
 *	none.
 * 
 * the formula for an ellipse through a meridional circle:
 * 
 *     2    2             2    2 
 *    r *cos (lat)       r *sin (lat)           
 *    ------------   +   ------------    =    1 
 *           2                  2               
 *        r                 r                  
 *         eq                pol
 * author: A. Porter
 * ----------------------------------------------------------------
 */

double 
navEarthRadius(celLatitude)
     double celLatitude;
{
  double c,r,s;

  c = cos(celLatitude);
  s = sin(celLatitude);
  r = 1./sqrt(c*c/(eqRad*eqRad) + s*s/(polRad*polRad));
  return r;  

}


/* ----------------------------------------------------------------
 * procedure navEarthRadCurv
 * input:
 *	geoCENTRIC (NB!) latitude;
 * results: 
 *	radius of curvature of the earth at this geocentric latitude;
 * side effects:
 *	none.
 *
 * source of the formula: Thomas_1953
 * ----------------------------------------------------------------
 */
double 
navEarthRadCurv( celLatitude )
     double celLatitude;
{
  double re2, rp2; 		/* dummies		*/
  double rho;			/* to hold curvature	*/ 
  double radius; 		/* local earth radius 	*/ 
  double r, z;			/* cylindrical coords	*/

  radius = navEarthRadius(celLatitude);
  r = radius*cos(celLatitude);
  z = radius*sin(celLatitude);

  re2 = eqRad*eqRad;
  rp2 = polRad*polRad;

  rho = sqrt(z*z*re2*re2 + r*r*rp2*rp2);
  rho = rho*rho*rho;
  rho = rho/(re2*rp2);
  rho = rho/(z*z*re2 + r*r*rp2);

  return rho;
}

 

/* ----------------------------------------------------------------
 * navOrbitSet sets the elements of an orbit
 * inputs
 *	orbital elements
 * 	9 of them! 
 * 	mass, e, T, A, O, i, o, refA, refT
 * outputs
 * 	pointer to an orbit
 * side effects
 * 	sets elements of that orbit
 * notes
 * 	if orb->o_refAnomaly is not consistent with orb->o_refTime,
 *	o_refAnomaly takes precedence
 * author: A. Porter
 * ----------------------------------------------------------------
 */

Orbit *
navOrbitSet( orb, mass, eccn, time, axis, node, incl, pgee, refA, refT )
     Orbit 	*orb;	/* if not null on input, this will be used */ 
     double 	mass;	/* mass of the central body (EARTH mass if zero on input) */
     double 	eccn;	/* eccentricity */
     double 	time;	/* perigee time */
     double 	axis;	/* semimajor axis */
     double 	node;	/* longitude of the ascending node */
     double 	incl;	/* inclination */
     double 	pgee;	/* argument of the perigee */
     double 	refA;	/* reference anomaly */
     double 	refT;	/* reference time */
{

    if (orb == NULL)
    {
	orb = (Orbit *) malloc(sizeof(Orbit));
	/* printf(" ptr to created orbit = %x \n", orb ); */
    }

    if (mass != 0.) 
      orb->o_mass	= mass;
    else
      orb->o_mass	= E_MASS;
    orb->o_eccen	= eccn;
    orb->o_periTime	= time;

    orb->o_axis		= axis;

    orb->o_er.er_1	= node;
    orb->o_er.er_2	= incl;
    orb->o_er.er_3	= pgee;
    orb->o_refAnomaly	= refA;
    orb->o_refTime	= refT;

    navResetPeriTime(orb);

    return orb;

}


/* ----------------------------------------------------------------
 * navInitialElements sets the elements of an orbit
 *	from starting parameters of convenience
 * inputs
 *	time, double 
 * 	position as a PCoord
 * 	direction; that is, angle of velocity 
 *		counter-clockwise from local parallel of latitude
 * outputs
 * 	pointer to an orbit
 * side effects
 * 	sets elements of that orbit
 * assumptions 
 * 	earth orbit
 *	perigee at starting position
 *	starting position is not directly over the pole (algebra bombs there)
 * author: A. Porter
 * ----------------------------------------------------------------
 */

Orbit *
navInitialElements( time, pos, theta, ecc, orb)
     double time; 	/* time, seconds */
     PCoord *pos;	/* geocentric position (radius from earth center, geocentric latitude) */
     double theta;	/* angle from local parallel of latitude to direction, */
                        /* counter-clockwise in N. hemisphere, 
			   clockwise in S. hemisphere */ 
     double  ecc;	/* requested eccentricity */
     Orbit *orb;	/* if not null on input, this will be used */ 
{
    Coord3d direction;	/* direction of motion implied by input specification theta */
    double   ct;		/* cos(theta) */
    double   st;		/* sin(theta) */
    Coord3d lat;	/* unit vector in direction of increasing latitude */
    Coord3d lng;	/* unit vector in direction of increasing longitude */
    Coord3d cpos;	/* position in cartesian coords */
    Coord3d pole;	/* unit vector perpendicular to the orbit */
    Coord3d nodevector;	/* unit vector in direction of the line of nodes */
    Coord3d solstice;	/* unit vector 90 degrees counter-clockwise 
			   from nodevector in orbit plane */ 
    double   node;	/* longitude of line of nodes */
    double   incl;	/* orbital inclination */
    double   pgee;	/* argument of the perigee */

    if (orb == NULL)
      orb = (Orbit *) malloc(sizeof(Orbit));

    orb->o_mass		= E_MASS;	/* assume this */
    orb->o_eccen	= ecc; 		/* assume circular orbit */
    orb->o_axis		= pos->pc_r / (1. - ecc);	/* perigee distance */
    orb->o_periTime	= time;		/* we define the present position 
					   to be the perigee; */
    orb->o_refTime	= time; 	/* thus refTime = periTime = now   */
    orb->o_refAnomaly	= 0.;		/* and refAnomaly = 0. */

    /* the node, perigee, and inclination could be found from the law of sines,
     * but the quadrants would be ambiguous, 
     * and they can be found from vector algebra unambiguously:
     */
    
    matPolarToRect( pos, &cpos); 
    GeoUnitVector( &cpos, &cpos);
    /* this will bomb if you try to start above the pole . . .: */
    lng.x = - cpos.y;
    lng.y =   cpos.x;
    lng.z =   0.;
    GeoUnitVector( &lng, &lng);

    lat.x = - cpos.x * sin(pos->pc_lat);
    lat.y = - cpos.y * sin(pos->pc_lat);
    lat.z =   cos(pos->pc_lat);
    GeoUnitVector( &lat, &lat);

    ct = cos(theta);
    st = sin(theta);
    direction.x = st * lat.x + ct * lng.x;
    direction.y = st * lat.y + ct * lng.y;
    direction.z = st * lat.z + ct * lng.z;

    GeoCrossProduct( &cpos, &direction, &pole);
    GeoUnitVector( &pole, &pole);

    node = PI * 0.5  +  atan2( pole.y, pole.x);
    incl = acos(pole.z);
    /* resolving quadrant ambiguity in the perigee is a little trickier */
    nodevector.x = cos(node);
    nodevector.y = sin(node);
    nodevector.z = 0.;
    GeoCrossProduct( &pole, &nodevector, &solstice );
    
    pgee = atan2( 
		 GeoDotProduct( &cpos, &solstice), 
		 GeoDotProduct( &cpos, &nodevector) 
		 );

    orb->o_er.er_1	= node;
    orb->o_er.er_2	= incl;
    orb->o_er.er_3	= pgee;

    return orb;

}

/* ----------------------------------------------------------------
 * testel tests Xelements()
 * no input, no output, prints lots 
 * explains as it goes
 * author: A. Porter
 * ----------------------------------------------------------------
 */

void
testxel()
{
    Orbit *orb;		/* to be filled in by navConvenientElements() */
    double latd_in;	/* inputs to navConvenientElements() */
    double longd_in;	/* etc. */
    double altd_in;	/* etc. */
    double latd_out;	/* to be derived from output of navConvenientElements() */
    double longd_out;	/* etc. */
    double altd_out;	/* etc. */
    int	  direction;	/* +1 = launch to north; -1 = launch to south */
    double eccentricity;	/* input to cnv.El.() */
    double toPerigee;	/* input to cnv.El.() */
    Coord3d velocity;	/* to be derived from output of navConvenientElements() */
    Coord3d position;	/* ditto */
    PCoord  rpos;	/* polar version of position */
    double time;	/* unix time, seconds past 1970.0 */
    double d;		/* days past 1990.0 */
    double inclination;	/* input to Conv.Elems() */
    int loop_counter;	/* counts times through the inner loop -- so we can stop early */
    int maxcount;	/* how many times through loop do we go? */
    double e_rad;	/* earth radius */ 
    Orbit *xorb;	/* returned by navConvenientElements() */
    double xx;		/* temp */
    double radius;	/* local earth radius */
    double gcl;		/* temp */
    double incl_out;	/* inclination out */
    double xlat;		/* diff between requested and delivered elements */
    double xlong;	/* ditto */ 
    double xincl;	/* ditto */
    double xalt;		/* ditto */
    
      

    d = 32.;
    time = calendarJulianToUnix( 2447891.5 + d );	/* jd 2447891.5 == 1990.0 */
    altd_in = 400000.;
    eccentricity = 0.;
    toPerigee = 0.;
      
    orb = (Orbit *) NULL;
    maxcount = 5000;
    loop_counter = 0;
    longd_in = -80.;
    latd_in = -80.;
    inclination = 80.;
    direction = 1; 

    while (1)
    {
	longd_in += 20.;
	longd_in = normalize( longd_in, 0., 360. );
	latd_in += 25.;
	latd_in = normalize( latd_in, -90., 90. );

	eccentricity += 0.001;
	if (eccentricity > 0.1) eccentricity = 0.;
	inclination += 10.;
	inclination = normalize( inclination, 0., 180. );
	direction *= -1;
	time += 100000.0; 
	altd_in += 5000.;
	if (altd_in > 1000000.) altd_in = 250000.;

	if (inclination < 90. && inclination  < ABS(latd_in) )
		      continue;
	if (inclination >= 90. && (180. - inclination)  < ABS(latd_in) )
		      continue;

	/* test Xelements */
	xorb = Xelements( time, latd_in, longd_in, altd_in, 
			 inclination, direction, orb );

	/* test navConvenientElements */
	if (xorb == NULL)
	{
	    continue;
	}

	++loop_counter;
	if (loop_counter > maxcount) goto quit_early;

	if (xorb->o_eccen > 0.0001)
	{
	    printf(" the orbit itself: \n");
	    dumpOrbit( xorb );
	}

	incl_out = xorb->o_er.er_2*TODEGREES ;

	navKepler( xorb, time, &position );
	matRectToPolar( &position, &rpos );

	latd_out     = navGeodetLat(rpos.pc_lat)*TODEGREES ;

	xx = rpos.pc_long - GAST( time ); 
	xx = normalize( xx, 0., PI2 )*TODEGREES;
	xx = normalize( xx, longd_in-180., longd_in+180. );
	longd_out = xx;

	gcl = rpos.pc_lat;
	radius = navEarthRadius( gcl );
	xx = rpos.pc_r - radius;
	altd_out = xx;

	xlat = latd_out - latd_in;
	xlong = longd_out - longd_in;
	xincl = inclination - incl_out;
	xalt = altd_out - altd_in;

	if (ABS(xlat) + ABS(xlong) + ABS(xincl) + ABS(xalt / altd_in) < 0.0001 ) 
	  continue; 

	printf("\n\n\n" );
	printf("\n inputs to Xelements(): \n");
	printf("    time  =  jd % 12.2f ", calendarUnixToJulian( time ) );
	printf("	i. e.,    %s; ", 
	       fvfJulianToDate( calendarUnixToJulian( time ) ) );  
	printf("    direction  %d \n", direction );
	printf(" requested lat % 12.2e long % 12.2e incl % 12.2e alt  % 12.2e \n", 
	       latd_in, longd_in, inclination, altd_in);
	printf("diff in    lat % 12.2e long % 12.2e incl % 12.2e alt  % 12.2e \n", 
	       xlat, xlong, xincl, xalt );
 

    }

quit_early:
    printf(" you decided to quit early \n" );
    exit(1);

}

      

/* ----------------------------------------------------------------
 * Xelements sets the elements of an orbit
 *	from starting parameters of convenience
 * inputs
 *	time, double float, in UNITS of seconds
 * 	latitude, longitude, altitude, orbital inclination
 * outputs
 * 	pointer to an orbit
 * side effects
 * 	sets elements of that orbit
 * assumptions 
 * 	earth orbit
 *	circular orbit
 * method
 *	vector analysis of the problem
 * units
 *	input is in DEGREES, METERS, SECONDS;
 *	internal calculations are in RADIANS
 * coordinate system: 
 *	orbit returned is with respect to the FIRMAMENT, NOT, repeat, NOT Greenwich !!!!
 *	equatorial coordinates of date; 
 *		i. e., we do not worry about precession or nutation,
 *	but we DO correct longitude for rotation of the earth 
 * author: A. Porter
 * ----------------------------------------------------------------
 */

Orbit *
Xelements( time, latitude, longitude, altitude, inclination, 
		      direction, orb )
     double time; 	/* time in double float format; zero of time is 1970.0 */
     double  latitude;	/* geoDETIC latitude at injection; units DEGREES */
     double  longitude;	/* EAST longitude at injection; units DEGREES  */
     double  altitude;	/* altitude, in METERS, at injection */	 	
     double  inclination; /* desired inclination of resulting orbit; units DEGREES  */
     int    direction;	/* direction of injection: >= 0  is to the North, < 0 is to the South */
     Orbit *orb;	/* if not null on input, this will be used */ 
{
    PCoord	rpos;
    Coord3d	position;
    Coord3d	velocity;
    Coord3d 	position1;
    Coord3d	velocity1;

    Coord3d	south;
    Coord3d	east;
    double	v_south;
    double	v_east;
    double 	radius;		/* earth radius here */
    double	geocentlat;	/* geos-centric latitude */
    double	thetaG;		/* longitude of greenwich in firmament sys.  */
    double 	f_long;		/* longitude of satellite in firmament system */
    double	speed;		/* speed for a circular orbit at this altitude */
    double 	trueAnomaly;	/* true anomaly at point of injection */
    double	xx;		/* temp */
    double	yy;		/* temp */
    int 	ok;		/* ok or not */

    ok = Xposvel( time, latitude, longitude, altitude, 
		 inclination, direction, &position, &velocity );
    if (ok == 0) return (Orbit *) NULL;

    latitude 	*= TORADIANS;
    longitude 	*= TORADIANS;
    inclination *= TORADIANS;
    if (direction >= 0) direction =  1;
    if (direction <  0) direction = -1;
    
    if ((inclination < PIhalf &&        inclination < ABS( latitude ) )
	||
       ( inclination > PIhalf && (PI - inclination) < ABS( latitude ) ) )
    {
	printf(" inclination must be at least as great as latitude of injection; \n");
	printf(" you asked for incl. %e, latitude %e \n", 
	       inclination*TODEGREES, latitude*TODEGREES );
	return (Orbit *) NULL;
    }

    if (latitude <= -89.7 || latitude >= 89.7 )
    {
	printf(" latitude out of range \n" );
	return (Orbit *) NULL;
    }
    longitude = normalize( longitude, 0., PI2 );
    
    if (!orb)
    {
	orb = (Orbit *) malloc(sizeof(Orbit));
    }

    navPosVelToOrbit( E_MASS, time, &position, &velocity, orb );

    return orb;

}

/* ----------------------------------------------------------------
 * Xposvel sets position and velocity, given latitude, longitude, 
 *	altitude and requested inclinitation of orbit;
 *	cf. Xelements()
 * inputs
 *	latitude, longitude, altitude, time, inclination, 
 *	pointers to position, velocity 
 * outputs
 *	1 for success, 0 for failure
 * side effects
 *	sets position, velocity
 * method
 *	cf. Xelements()
 * author: A. Porter
 * ----------------------------------------------------------------
 */

int
Xposvel( time, latitude, longitude, altitude, inclination, direction, position, velocity )
     double time; 	/* time in double float format; zero of time is 1970.0 */
     double  latitude;	/* geoDETIC latitude at injection; units DEGREES */
     double  longitude;	/* EAST longitude at injection; units DEGREES  */
     double  altitude;	/* altitude, in METERS, at injection */	 	
     double  inclination; /* desired inclination of resulting orbit; units DEGREES  */
     int    direction;	/* direction of injection: 
			   >= 0  is to the North, < 0 is to the South */
     Coord3d *position;	/* where to put the calculated position */
     Coord3d *velocity; /* where to put the calculated velocity */
{
    PCoord	rpos;
    Coord3d	south;
    Coord3d	east;
    double	v_south;
    double	v_east;
    double 	radius;		/* earth radius here */
    double	geocentlat;	/* geos-centric latitude */
    double	thetaG;		/* longitude of greenwich in firmament sys.  */
    double 	f_long;		/* longitude of satellite in firmament system */
    double	speed;		/* speed for a circular orbit at this altitude */
    double 	trueAnomaly;	/* true anomaly at point of injection */

    latitude 	*= TORADIANS;
    longitude 	*= TORADIANS;
    inclination *= TORADIANS;
    if (direction >= 0) direction =  1;
    if (direction <  0) direction = -1;
    
    if ((inclination < PIhalf &&        inclination < ABS( latitude ) )
	||
       ( inclination > PIhalf && (PI - inclination) < ABS( latitude ) ) )
    {
	printf(" inclination must be at least as great as latitude of injection; \n");
	printf(" you asked for incl. %e, latitude %e \n", 
	       inclination*TODEGREES, latitude*TODEGREES );
	return 0; 
    }

    if (latitude <= -89.7 || latitude >= 89.7 )
    {
	printf(" latitude out of range \n" );
	return 0; 
    }
    longitude = normalize( longitude, 0., PI2 );
    
    /* the geocentric latitude: */
    geocentlat = navGeocentLat( latitude );

    /* the longitude of Greenwich */

    thetaG	= GAST( time ); 
    f_long	= thetaG + longitude;

    radius		= navEarthRadius( geocentlat ); 

    /* now find the velocity, assuming circular orbit and desired inclination */
    /* v dot r = 0; velocity is perpendicular to radius */
    /* z-unit dot ( r cross v) = cos(inclination)       */
    /* work in the south-east-zenith coord system       */

    v_east 	= cos( inclination) / cos( latitude );
    v_south	= - direction * sqrt( 1.0 - v_east*v_east );
    
    /* obtain velocity in xyz coord system */ 

    east.x 	= - sin( f_long );
    east.y	=   cos( f_long );
    east.z	=   0.;

    south.x	=   cos( f_long) * sin( geocentlat ); 
    south.y	=   sin( f_long) * sin( geocentlat ); 
    south.z	= - cos( geocentlat );

    GeoZeroVector( velocity );
    matAccumulateVector( v_east,  &east,  velocity );
    matAccumulateVector( v_south, &south, velocity );
    
    rpos.pc_r	= altitude + radius;
    rpos.pc_lat = geocentlat;
    rpos.pc_long = f_long;
    matPolarToRect( &rpos, position );

    speed 	= sqrt(MU/rpos.pc_r);
    GeoUnitVector( velocity, velocity );
    GeoScalarMultVector( speed, velocity, velocity );
    if ( direction * velocity->z < 0 ) 
      GeoScalarMultVector( -1.0, velocity, velocity );

    return 1;
}


/* ----------------------------------------------------------------
 * takes parameters of lambert's problem and the proposed solution,
 *      finds orbit through first pos/vel, calculates second pos/vel, 
 *	and compares to proposed 2nd pos/vel;
 *      and it prints results and figures of merit 
 *
 * ----------------------------------------------------------------
 */ 


double 
testSolution( fo, mass, r1, r2, v1f, v2f, t1, t2 )
     FILE	*fo;	/* where to write to */ 
     double 	mass;	/* central body mass */ 
     Coord3d	*r1;	/* the given r1 */ 
     Coord3d	*r2;	/* the given r2 */ 
     Coord3d	*v1f;	/* the proposed v1 */ 
     Coord3d	*v2f;	/* the proposed v1 */ 
     double 	t1;	/* initial time */ 
     double 	t2;	/* final   time */ 
{
    double 	transferTime;	/* the sought / known transfer time */ 
    Coord3d	r2f;	/* the found r2 */ 
    Coord3d	v2fx;	/* the found v2 */ 
    Coord3d	v2diff;	/* diff of v2fx calc here and v2f supplied by calling routine */ 
    Coord3d	tempvec; 

    double	temp; 
    double	deltaT;		/* error in transfer time */ 
    double	deltaR;		/* miss distance */ 
    Orbit 	orb; 
    double	trueAnomaly1;
    double	trueAnomaly2; 
    double 	transferAngle;
    double 	setAngle;	/* the set transfer angle, w/o regard to direction */ 
    double 	period;
    int		revs;
    int		shortway;	/* whether the transfer is short way or not */ 
    double	fulltheta;	/* full transfer angle, mult-revs included */ 

    double 	dr, dt;		/* dr/r, dt/t, i.e., frac errors */ 
    double 	error=0.;	/* sqrt( dr*dr + dt*dt ) */ 

    transferTime = t2 - t1; 
    setAngle = matAngleofVectors( r1, r2 ); 

    navPosVelToOrbit( mass, t1, r1, v1f, &orb ); 

    /* find true anomalies: */  
    trueAnomaly1 = navTrueAnomaly( &orb, t1 ); 
    trueAnomaly2 = navTrueAnomaly( &orb, t2 ); 
    transferAngle = normalize(trueAnomaly2, trueAnomaly1, trueAnomaly1 + PI2 )
	                    - trueAnomaly1; 

    /* final position and final velocity from the orbit of initial conditions */ 
    navAnomalyToPosition( &orb, trueAnomaly2, &r2f );
    navVelocity( &orb, trueAnomaly2, &v2fx );

    /* determine the full transfer angle, mult-revs included */ 
    fulltheta = transferAngle; 
    if ( ellipse( &orb ) ) 	/* full/mult revs are possible  */ 
    {
	period = PI2 / navMeanMotion( &orb ); 
	revs = (int) (transferTime / period); 
	fulltheta = transferAngle + revs * PI2; 
    }

    /* determine how close the orbit comes to desired r2 at t2 */ 
    fprintf( fo, "\n"); 
    GeoSubtractVector( r2, &r2f, &tempvec ); 
    temp = GeoVectorLength( &v2fx ); 
    deltaT = GeoDotProduct( &tempvec, &v2fx ) / (temp*temp); 
    dt = ABS( deltaT ) / transferTime; 

    fprintf( fo, "                set transfer angle = % f degs \n", setAngle * TODEGREES ); 
    fprintf( fo, " signed/directional transfer angle = % f degs \n", transferAngle * TODEGREES ); 
    fprintf( fo, "               full transfer angle = % f degs \n", fulltheta * TODEGREES ); 
    if ( orb.o_eccen < 1.0 )
      fprintf( fo, "               number of full revs = %d \n", revs ); 
    fprintf( fo, " error in transfer time % e,       fraction error  % e \n", 
	   deltaT, dt ); 
    deltaR = 0.; 
    if ( GeoVectorLength( &tempvec ) != 0. )
    {
	deltaR = GeoVectorLength( &tempvec ) * sin( matAngleofVectors( &tempvec, &v2fx ) ); 
    }
    dr = ABS( deltaR ) / GeoVectorLength( r2 );
    fprintf( fo, "                                  fractional miss distance = % e \n", dr ); 

    /* print out merits of the proposed solution to the lambert-problem */ 
    fprintf( fo, " found v1: % e, % e, % e \n", v1f->x, v1f->y, v1f->z ); 
    /* fprintf( fo, " found v2: % e, % e, % e \n", v2f->x, v2f->y, v2f->z ); */
    /* fprintf( fo, " calc  v2: % e, % e, % e \n", v2fx.x, v2fx.y, v2fx.z ); */
    GeoSubtractVector( &v2fx, v2f, &tempvec );
    fprintf( fo, " diff, calc - found v2:     % e, \n", GeoVectorLength( &tempvec ) ); 
    fprintf( fo, "found eccentricity %f, axis %f AU \n", orb.o_eccen, orb.o_axis/AU ); 

    fflush( fo ); 
    if ( dr > -1.e20 && dr < 1.e20 && dt > -1.e20 && dt < 1.e20 )
      error = sqrt( dr*dr + dt*dt ); 
    else
      error = 1.e100; 

    fprintf( fo, "                error returned     = %e \n", error  ); 

    return error;  

} 
