
/*
 * navMain.c
 * 
 * the main Keplerian orbit calculation routines 
 * 
 * Copyright (c) 1987-1994 Andrew P. Porter
 */


extern double sin(), cos(), tan(), asin(), acos(), atan(), atan2(), sqrt();

#include <stdio.h>
#include <math.h>

#include "../include/ms.h"
#include "../include/matrix.h"
#include "../include/hnav.h" 




/*
 * ---------------------------------------------------------------- 
 * navAnomalyToTime finds time, given true anomaly
 * inputs
 *	orbit
 *	true anomaly
 * outputs
 *	time
 * side effects
 *	none
 * method
 *	compare navPerigeeTime
 * ----------------------------------------------------------------
 */

double
navAnomalyToTime( orb, trueAnomaly )
     Orbit *orb;
     double trueAnomaly;
{
    double ecc;		/* eccentricity */
    double meanMotion;
    double eccentricAnomaly;
    double meanAnomaly;
    double period;
    double ct, st; 	/* trig fns of true anomaly */
    double ce, se;	/* trig fns of ecce anomaly */ 
    double time;	/* the quantity to be found: time at given true anomaly */
    double temp; 

    ecc = orb->o_eccen;
    meanMotion = navMeanMotion(orb);

    if ( ellipse( orb )  )
    {
	ct = cos(trueAnomaly);
	st = sin(trueAnomaly);
	ce = (ecc + ct) / ( 1. + ecc*ct);
	se = st * sqrt(1. - ecc*ecc) / (1. + ecc*ct);
	eccentricAnomaly = atan2(se,ce);

	meanAnomaly = eccentricAnomaly - ecc*se;
	meanAnomaly = normalize(meanAnomaly,0.,2.*PI);

	time = orb->o_periTime + meanAnomaly / meanMotion;

	period = 2.*PI / meanMotion; 
	time = normalize(time, 0., period);

	return time;
    }
    else if ( hyperbola( orb ) )
    {
	eccentricAnomaly = navTrueToEccentricAnomaly( ecc, trueAnomaly ); 
	meanAnomaly = navEccToMeanAnomaly( ecc, eccentricAnomaly ); 
	time = orb->o_periTime + meanAnomaly / meanMotion;
	return time; 
    }
    else if ( parabola( orb ) )
    {  
	meanAnomaly = parabolicTrueToMeanAnomaly( trueAnomaly );
	time = orb->o_periTime + meanAnomaly / parabolicRate( orb );
	return time; 
    }
    else
    {
	printf(" in navAnomalyToTime(); something is wrong, ecc = %e \n", ecc ); 
	return 0.;
    }

}


/*
 * ---------------------------------------------------------------- 
 *
 * navKepler --
 * navPosition --
 * inputs: 
 *	pointer to an orbit, 
 *	time, 
 *		double precision double (navKepler)	 
 *		double integer	       (navPosition)
 *	pointer to a Coord3d position for the result
 *
 * input:
 *	orbit, sidereal time;
 *	pointer to PCoord, to store result in
 * results:
 * 	none;
 *
 * side effects:
 *	puts position coords in *pc 
 *
 * the method:
 * 	get nu, true anomaly, from navTrueAnomaly()
 *	get position from navAnomalyToPosition()
 *
 * ---------------------------------------------------------------- 
 */

void
navKepler( orbit, time, pos ) 
     Orbit *orbit;
     double time;	/* time for which position is requested	*/
     Coord3d *pos;
{
  double trueAnomaly;	

  trueAnomaly = navTrueAnomaly( orbit, time ); 
  navAnomalyToPosition( orbit, trueAnomaly, pos );

}


/* ----------------------------------------------------------------
 * navOrbitToPosVel( orbit, time, pos, vel )
 * inputs
 * 	time, ptr to orbit, ptrs to pos, vel
 * outputs
 *	none  
 * side effects
 * 	puts position and velocity where the pos, vel ptrs point to
 * method
 *	calls navTrueAnomaly, navAnomalyToPosition, navVelocity 
 * ----------------------------------------------------------------
 */ 

void  
navOrbitToPosVel( orbit, time, pos, vel )
     Orbit *orbit;
     double time;
     Coord3d *pos;
     Coord3d *vel;
{
    double trueAnomaly; 

    trueAnomaly = navTrueAnomaly( orbit, time ); 
    navAnomalyToPosition( orbit, trueAnomaly, pos );
    navVelocity( orbit, trueAnomaly, vel );
  
}



/* ----------------------------------------------------------------
 * given an orbit and a true anomaly, find position
 * inputs
 *	orbital elements
 *	true anomaly
 *	pointer to position in which to store results 
 * outputs
 *	none
 * side effects
 *	sets *pos
 * ----------------------------------------------------------------
 */

void
navAnomalyToPosition( orbit, trueAnomaly, pos )
  Orbit 	*orbit;
  double 	trueAnomaly;
  Coord3d 	*pos;
{
  double 	axis; 		/* semi-major axis			*/
  double 	p;		/* semi-latus rectum			*/
  double 	r;		/* scalar radius;  = p/(a + e*cos(trueAnomaly))	*/
  matrix3d 	et, etx;	/* rot matrix from firmament to orbit   */
  Coord3d  	rr;		/* position in orbit's frame		*/
  Coord3d  	position;	/* position in firmament 		*/
  double 	ecc;		/* eccentricity */
  double 	eccAnomaly;	/* eccentric anomaly */ 


  ecc = orbit->o_eccen;
  axis = orbit->o_axis;

  if ( ellipse( orbit)  )		/* the elliptic case */ 
  {
      p = axis*(1. - ecc*ecc);
  }
  else if ( hyperbola( orbit )  )		/* the hyperbolic case */ 
  {
      p = axis*(1. - ecc*ecc);
  }
  else if ( parabola( orbit ) )				/* the parabolic case */ 
  {
      p = axis;  /* yrrch!  we use orb->o_axis for the semi-parameter for a parabola */ 
  }

  r = p/(1. + ecc*cos(trueAnomaly)); 
  rr.x =   r*cos(trueAnomaly);  /* cartesian coords in the orbit's frame */
  rr.y =   r*sin(trueAnomaly);	
  rr.z =   0.0;

  /*  et = matrix from orbit euler angles; 	   */
  matEulerToMat(&orbit->o_er,et);

  /* position = matrix rot * vector rr: 			*/
  matMatVV( et, &rr, &position);

  pos->x = position.x;
  pos->y = position.y;
  pos->z = position.z;

}



/* ----------------------------------------------------------------
 * navTrueAnomaly
 * 	called by navKepler et al, 
 *	finds true anomaly
 *	given orbit and time
 * method:
 * 	n = mean motion = sqrt(mu/a*a*a)
 * 	m = mean anomaly = n*(t-T)
 * 	E = eccentric anomaly;
 * 	e = eccentricity
 * 	solve m = E - e*sin(E) for E
 * 		iterate:
 * 		E0 = M0 = M
 * 		En+1 = En + (M - Mn)/(1 - e*cos(En))
 *		Mn+1 = En+1 - e*sin(En+1)
 * 	nu = true anomaly
 * 	cos(nu) = (e - cos(E))/(ecos(E) - 1)
 * inputs
 *	orbital elements
 *	time
 * outputs
 *	true anomaly
 * side effects
 *	none
 * ----------------------------------------------------------------
 */


double 
navTrueAnomaly( orbit, time )
     Orbit *orbit;		/* orbital elements */
     double time;		/* time for which true anomaly is sought */
{
  double  	meanAnomaly;	/* mean anomaly; = n*(t-T)		*/
  double  	eccAnomaly;	/* eccentric anomaly 			*/
  double  	trueAnomaly;	/* true anomaly				*/


  if ( !parabola( orbit )  ) 
  {	/* (the orbit is safely elliptic or hyperbolic) */ 
      meanAnomaly 	= navMeanAnomaly( orbit, time );
      eccAnomaly 	= navMeanToEccentricAnomaly (orbit->o_eccen, meanAnomaly);
      trueAnomaly 	= navEccentricToTrueAnomaly (orbit->o_eccen, eccAnomaly);
      return trueAnomaly;
  }
  else 		/* (the orbit is close enough to treat as parabolic) */ 
  {
      trueAnomaly = barkersEquation( orbit, time ); 
      return trueAnomaly; 
  }

}


/* ----------------------------------------------------------------
 * navMeanAnomaly
 * inputs
 *	orbit, time
 * outputs
 * 	mean anomaly
 * side effects
 *	none
 * ----------------------------------------------------------------
 */

double
navMeanAnomaly( orbit, time )
     Orbit *orbit;
     double time;
{
  double mu;			/* G*M */
  double axis; 		/* semi-major axis			*/
  double meanMotion; 	/* mean motion; = sqrt(mu/a*a*a) 	*/
  double meanAnomaly;	/* mean anomaly; = n*(t-T)		*/
  double reduced_time;  /* time with excess orbital periods removed */
  double period;	/* orbital period 			*/

  axis = orbit->o_axis;
  mu = BIG_G * orbit->o_mass;

  /* for ellipse/hyperbola:  */ 
  meanMotion = sqrt(mu/(  ABS(axis*axis*axis) )); /* for hyperbolic orbits, axis < 0. */
  /* see Battin, BMMA, p. 150; <axis> here is semi-latus-rectum */ 
  if (parabola( orbit ))
    meanMotion *= 4.;

  reduced_time = time - orbit->o_periTime;

  /* correct to last periapse passage */ 
  if ( ellipse( orbit ) ) 	/* elliptic case requires period reduction */ 
  {
      period = 2.*PI / meanMotion;
      reduced_time = normalize( reduced_time, 0., period );
  }
  meanAnomaly = meanMotion * reduced_time;

  return meanAnomaly;
}


/* ----------------------------------------------------------------
 * navMeanToEccentricAnomaly
 * 	finds eccentricAnomaly, from mean anomaly and eccentricity
 * inputs
 *	eccentricity, mean anomaly
 * outputs
 * 	eccentric anomaly
 * side effects
 *	none
 * iterative method from BMW, p. 221-222; 
 * compare Green, p. 145, eqn. (6.35);    
 * compare BMMA, p. 217, prob. 5-24, and p. 194, eqn. (5.4). 
 * ----------------------------------------------------------------
 */

double
navMeanToEccentricAnomaly( ecc, meanAnomaly )
  double ecc;		/* eccentricity */
  double meanAnomaly;
{
    double eo;		/* old guess at eccentric anomaly	*/
    double en;		/* new guess at eccentric anomaly	*/
    double mo;		/* old guess at mean anomaly		*/
    double mn;		/* new guess at mean anomaly		*/
    int i;		/* loop counter				*/
    double  eccAnomaly;	/* eccentric anomaly 			*/


    if ( ecc < 0. )
    {
	printf(" negative eccentriticy in navMeanToEccentricAnomaly \n" );
	exit(1);
    }

    if ( ecc < 1.0 - PARACRIT )	/* elliptical orbit */ 
    {
	/* Odell and Gooding's short routine: */ 
	eccAnomaly = ekepl1( meanAnomaly, ecc ); 
	return eccAnomaly;
    }

    if ( ecc > 1.0 + PARACRIT  )	/* hyperbolic orbit */ 
    {
	/* the method of Gooding and Odell, Cel Mech 44 (1988) 267, p. 279-280: */
	eccAnomaly = asinh( shkepl(  (meanAnomaly/ecc),    (1. - 1./ecc)   )   ); 
	if ( ABS(meanAnomaly - ecc*sinh(eccAnomaly) + eccAnomaly) 
	      > ABS(meanAnomaly)*1.e-12 
	    )
	{
	    printf(" dubious convergence hyperbolic anomaly: \n" );
	    printf(" mean anomaly %e, ecc*sinh(eccAnom) - eccAnom %e, diff %e \n",
		   meanAnomaly, ecc*sinh(eccAnomaly) - eccAnomaly, 
		   meanAnomaly - ecc*sinh(eccAnomaly) - eccAnomaly ); 
	}	
	return eccAnomaly; 
    }

    else 		/* parabola (forbidden) */ 
    {
	printf(" you called navMeanToEccentricAnomaly() for a parabolic orbit! \n"); 
	return 0.;
    }
    
}



/* ----------------------------------------------------------------
 * ekepl1 is the first of Odell and Gooding's routines;
 * inputst 
 *	meanAnomaly
 *	eccentricity 
 * outputs
 *	eccentric anomaly 
 * side effects 
 * 	none 
 * method
 *	see A. W. Odell, R. H. Gooding, Cel Mech 38 (1986) 307-334
 *	legendre-based starter, halley iterator 
 * ----------------------------------------------------------------
 */

double 
ekepl1( meanAnomaly, ecc )
     double meanAnomaly;
     double ecc; 		/* eccentricity */ 
{
    double testsq=1.e-8, c, s, psi; 
    double xi, eta, fd, fdd, f; 

    c = ecc * cos( meanAnomaly );
    s = ecc * sin( meanAnomaly ); 
    psi = s / sqrt( 1. - c - c + ecc*ecc ); 

loop:
    xi = cos( psi );
    eta = sin( psi );
    fd = (1. - c*xi) + s*eta;
    fdd = c*eta + s*xi;
    f = psi - fdd; 
    psi -= f*fd / (fd*fd - 0.5*f*fdd );
    if ( f*f >= testsq ) 
      goto loop;

    return meanAnomaly + psi; 

}




/* ----------------------------------------------------------------
 * shkepl is a c translation of the routine of the same name in 
 *	Gooding and Odell, Cel Mech 44 (1988) 267, p. 279-280 
 * inputs
 *	meanAnomaly / eccentricity
 *	1 - 1/eccentricity
 * outputs
 *	sinh( eccentric anomaly )
 * side effects 
 * 	none 
 * method 
 *	see comments and fortran code in their paper 
 * ----------------------------------------------------------------
 */ 

double 
shkepl( el, g1 )
     double el;		/* mean anomaly / eccentricity */ 
     double g1;		/* 1. - 1./eccentricity */ 
{
    double sw=0.5;
    double ahalf=0.5;
    double asixth=0.16666666666666666666666666;
    double athird=0.33333333333333333333333333;

    double s, g, cl, al, w, f, fd, fdd, fddd, stemp; 
    double s0, s1, s2, s3, ds; 
    int iter; 

    if (el == 0.)
    {
	return 0.;
    }
    s = el; 

    g = 1. - g1;
    cl = sqrt( 1. + el*el );
    al = asinh( el );
    w = (g*g * al) / ( cl*cl*cl );
    s = 1. - g/cl;
    s = el + g*al / cuberoot( s*s*s + w*el*(1.5 - g/0.75) ); 

    for( iter=0; iter<2; ++iter)	/* two iterations max */ 
    {
	s0 = s*s;
	s1 = s0 + 1.0;
	s2 = sqrt( s1 );
	s3 = s1*s2; 

	fdd = g*s/s3;
	fddd = g * ( 1. - 2.*s0 ) / ( s1*s3 ); 
	if ( s0*asixth + g1 >= sw )
	{
	    f = s - g*asinh(s) - el;
	    fd = 1. - g/s2;
	}
	else
	{
	    f = shmkep( g1, s ) - el;
	    fd = ( s0/(s2 + 1.) + g1) / s2;
	}

	ds = f*fd / ( ahalf*f*fdd - fd*fd ); 
	stemp = s + ds;

	if (stemp == s) break;
	
	f  +=  ds * (fd + ahalf*ds*(fdd + athird*ds*fddd) );
	fd +=  ds * (fdd + ahalf*ds*fddd); 
	s = stemp - f/fd; 

    }

    return s; 

}


#ifdef extendedcomment 
/* this is old code that solves Kepler's equation 
 * by a simple brute-force iteration 
 */

        /* elliptic case: */ 
	/* this is the old procedure written by A. Porter */ 
	mn = meanAnomaly; 
	en = mn;	/* initial guess for ecc. anomaly	*/
        /* BMMA, eq (5.4): */ 
	en += ecc * sin( mn ) / ( 1.0 - sin( mn + ecc )  +  sin( mn ) );  
	for (i=0; i<100; ++i) {
	    eo = en;
	    mo = eo - ecc*sin(eo);
	    if (ABS(mo - meanAnomaly) <= 1.e-14) break;
	    en = eo + (meanAnomaly - mo)/(1. - ecc*cos(eo));
	}
	if (i >= 99) {
	    printf("elliptic eccentric anomaly did not converge;\n");
	    printf("i, meanAnomaly, eo, en, mo, mn =  %d %e,  %e,  %e,  %e,  %e\n",
		   i, meanAnomaly, eo, en, mo, mn);
	}
	eccAnomaly = en;

        /* hyperbolic case: */ 
        /* this method works, but takes MANY iterations; from Battin, BMMA: */ 
        mn = meanAnomaly; 
	if ( ABS(mn) > 1.e-12 ) 
        {
          /* BMMA, eq (5.5): */ 
	  en = mn * mn  /  ( ecc*(ecc-1.0) * sinh(mn/(ecc-1.0)) - mn );	
        }
	else
	  en = meanAnomaly / (ecc - 1.0); 

	for (i=0; i<100; ++i)
	{	/* this works, but makes VERY slow convergence for low mean anomaly; 
		   gets better for high */ 
	    eo = en;
	    mo = ecc * sinh(eo) - eo;
	    if (ABS(mo - meanAnomaly) <= 1.e-12) break;
	    en = asinh( (meanAnomaly + eo) / ecc ); /* BMMA, p. 198, prob. 5-8. */ 
	}
	if (i >= 99) 
	{
	    printf("hyperbolic eccentric anomaly did not converge;\n");
	    printf("i %d, ecc %e, meanAnomaly %e, eo %e, en %e, mo %e \n", i, ecc, meanAnomaly, eo, en, mo);
	}
	eccAnomaly = en;
	return eccAnomaly;
#endif extendedcomment 

/* ----------------------------------------------------------------
 * shmkep is called by shkepl; see gooding and odell, cited above
 * inputs
 *	g1, i. e., (1 - 1/e)
 *	S,  i. e., ???? 
 * outputs 
 *	???? 
 * side effects
 * 	none 
 * method 
 * 	see their listing 
 * ----------------------------------------------------------------
 */ 

double shmkep( g1, s )
     double g1;
     double s;
{
    double g, t, tsq, x, term, twoi1, x0; 

    g = 1. - g1; 
    t = s / ( 1. + sqrt(1. + s*s ) ); 
    tsq = t*t;
    x = s * (g1 + g*tsq); 

    term = 2. * g * t;
    twoi1 = 1.;

    /* yes, a wretched fortran kludged loop: */ 
one:	
    twoi1 += 2.0; 
    term = term * tsq;
    x0 = x;
    x -= term/twoi1; 
    if ( x != x0 ) goto one;

    return x; 

} 


double 
cuberoot( x ) 
     double x;
{
    return exp( log( x ) / 3. );
}



/* ----------------------------------------------------------------
 * navEccentricToTrueAnomaly 
 *	given eccentricity and eccentric anomaly
 * inputs
 *	eccentricity, eccentric anomaly 
 * outptus
 *	true anomaly
 * side effects
 * 	none
 * true anomaly from eccentric anomaly, from BMW (4.2-12, 4.2-13), p. 186: 
 * hyperbolic case, BMMA, eq (4.56) 
 * ----------------------------------------------------------------
 */

double 
navEccentricToTrueAnomaly( ecc, eccAnomaly )
  double ecc;		/* eccentricity */
  double eccAnomaly;   	/* eccentric anomaly */
{
    double zx, zy, zz;		/* temps */
    double trueAnomaly;

    if ( ecc < 1.0 )	/* elliptical orbit */ 
    {
	zx = (ecc - cos(eccAnomaly))/(ecc*cos(eccAnomaly) - 1.);
	zy = sqrt(1. - ecc*ecc)*sin(eccAnomaly) / (1. - ecc*cos(eccAnomaly));
	trueAnomaly = atan2(zy,zx);
	return trueAnomaly;
    }
    if (ecc > 1.0 )	/* hyperbolic orbit */ 
    {
	zz = tanh( 0.5 * eccAnomaly )  *  sqrt( (ecc + 1.0) / (ecc - 1.0) );
	trueAnomaly = 2.0 * atan( zz ); 
	return trueAnomaly; 
    }
    else 		/* parabola (forbidden) */ 
    {
	printf(" you called navEccentricToTrueAnomaly() for a parabolic orbit! \n"); 
	return 0.;
    }
    
}

/* ----------------------------------------------------------------
 * navTrueToEccentricAnomaly(ecc,trueAnomaly)
 * 	is inverse of navEccentricToTrueAnomaly
 * inputs
 *	eccentricity
 *	true anomaly
 * outputs
 *	eccentric anomaly
 * side effects
 *	none
 * method
 *	sin, cos of ecc. anomaly from formulas 
 *	BMW, p. 185 (4.2-8), 
 *	BMW, p. 187 (4.2-13), or
 *	NASA Orbital Flight Handbook, v. 1.1, p. III-21
 * hyperbolic method: BMMA, eq (4.56) 
 * ----------------------------------------------------------------
 */

double
navTrueToEccentricAnomaly( ecc, trueAnomaly )
     double ecc;		/* eccentricity */
     double trueAnomaly;	/* true anomaly */
{
    double sinE, cosE;	/* trig fns of ecc anomaly */
    double temp; 

    if ( ecc < 1.0 )	/* elliptical orbit */ 
    {
	sinE = sqrt(1. - ecc*ecc) * sin(trueAnomaly) / (1. + ecc * cos(trueAnomaly));
	cosE = (ecc + cos(trueAnomaly)) / (1. + ecc * cos(trueAnomaly));
	return atan2( sinE, cosE );
    }
    if (ecc > 1.0 )	/* hyperbolic orbit */ 
    {
	temp = tan( 0.5 * trueAnomaly )  *  sqrt( (ecc - 1.0) / (ecc + 1.0) ); 
	if (temp > 1. )
	{
	    printf(" bad arg to atanh %e, 1-arg %e, in navTrueToEccentricAnomaly \n", 
		   temp, 1. - temp );
	}
	return 2.0 * atanh( temp ); 
    }
    else 		/* parabola (forbidden) */ 
    {
	printf(" you called navTrueToEccentricAnomaly() for a parabolic orbit! \n"); 
	return 0.;
    }

}


/* ----------------------------------------------------------------
 * navEccToMeanAnomaly(ecc,eccentricAnomaly)
 *	finds mean anomaly
 * inputs
 *	eccentricity
 *	eccentric anomaly
 * outputs
 * 	mean anomaly
 * side effects
 *	none
 * method
 *	M = E - e*sinE -- simple 
 *	N = e*sinhH - H, for hyperbolas; BMMA, eq (4.58)  
 * ----------------------------------------------------------------
 */

double navEccToMeanAnomaly( ecc, eccAnomaly )
     double ecc;
     double eccAnomaly;
{
    if ( ecc < 1.0 ) 
      return eccAnomaly - ecc * sin(eccAnomaly);
    if ( ecc > 1.0 ) 	
      return ecc * sinh( eccAnomaly ) - eccAnomaly; 
    else 
    {
	printf(" you called navEccToMeanAnomaly() for a parabolic orbit! \n"); 
	return 0.;
    }

}








/* ----------------------------------------------------------------
 * navVelocity
 *	calculates cartesian vector velocity
 *	given orbital elements and time for which velocity is sought
 * method
 *	components of velocity in BMW, p. 73, eqn. (2.5-4), 
 *	in PQW system, that is, in orbit coord system,
 *	then transposed to world system
 * inputs
 *	orbit
 *	time
 *	pointer to Coord3d to hold velocity
 * outputs
 *	none
 * side effects
 *	sets velocity
 * ----------------------------------------------------------------
 */
 
void
navVelocity( orbit, trueAnomaly, velocity)
  Orbit *orbit;
  double trueAnomaly;	
  Coord3d *velocity;
{
  double mu;		/* G*M */
  double parameter;	/* semi-latus-rectum */
  double scaleFactor;	/* sqrt(mu/parameter) */
  double ecc;		/* eccentricity */
  Coord3d vv; 		/* velocity in orbit's system */ 
  matrix3d mat, matx;	/* transfrom from orbit to world */
  double semiMajorAxis;
  double semiMinorAxis;
  double meanMotion; 
  double eccAnomaly;
  double eccAdot;	/* time derivative of eccAnomaly */ 
  double thetaInfinity;	/* see Dave Sonnabend's formulas, his pp.7-8 */ 
		
  double vmag;		/* magnitude of v, used in parabolic case */ 
  double rr;		/* length of r, used in parabolic case */ 

  mu = orbit->o_mass*BIG_G;
  ecc = orbit->o_eccen;

  if ( ellipse( orbit )  )		/* the elliptic case */ 
  {
      parameter = orbit->o_axis * (1. - ecc*ecc);
      scaleFactor = sqrt(mu / parameter);

      vv.x = - scaleFactor * sin(trueAnomaly);
      vv.y =   scaleFactor * (ecc + cos(trueAnomaly));
      vv.z = 0.;
  }
  else if ( hyperbola( orbit )  )
  {	
      eccAnomaly = navTrueToEccentricAnomaly( ecc, trueAnomaly ); 
      meanMotion = navMeanMotion( orbit ); 
      eccAdot = meanMotion / ( ecc * cosh( eccAnomaly ) - 1.0 ); 
      /* printf("   hyperbolic velocity: \n" );  */ 
      /* printf("     ecc-A %e, meanMotion %e, eccAdot %e \n", 
	 eccAnomaly, meanMotion, eccAdot ); */ 

      /* careful of signs . . . */ 
      semiMajorAxis = orbit->o_axis; 
      /* Dave Sonnabend, p. 8; BMMA p. 166, prob 4-14: */
      thetaInfinity = atan( sqrt ( (ecc*ecc - 1.0) )  );  	
      semiMinorAxis = -semiMajorAxis * tan( thetaInfinity ); 	/* Sonnabend, p. 8 */ 
      /* printf("     a %e, b %e, theta-inf %f \n", 
	 semiMajorAxis, semiMinorAxis, thetaInfinity*TODEGREES ); */ 

      /* differentiating BMMA, eq (4.52): */ 
      vv.x = semiMajorAxis * sinh( eccAnomaly) * eccAdot ;
      vv.y = semiMinorAxis * cosh( eccAnomaly) * eccAdot ;
      vv.z = 0.; 

      /* dumpVec( &vv, "velocity" ); */ 
  }
  else if ( parabola( orbit ) )
  {
      /* the parabolic case: things get simpler */ 
      /* magnitude of v is given by v*v/2 = mu/r */ 
      /* in nav.h, for ecc = 1., the Orbit structure has semi-parameter, not axis, 
       * in the place of the semi-major axis . . . 
       */ 
      rr = orbit->o_axis / (1. + cos(trueAnomaly) );
      vmag = sqrt( 2.*mu/rr ); 

      if (vmag == 0.)
      {
	  printf(" in navVelocity, bad vmag == 0.; \n" );
	  printf("                     axis %e, rr %e, mu %e, true anomaly %f \n",
		 orbit->o_axis, rr, mu, trueAnomaly ); 
      }

      /* flyout angle == 1/2 trueAnomaly: BMMA, p. 156; 	
       *	 juggle angles to get the following: 
       */ 
      vv.x = - vmag * sin( 0.5*trueAnomaly );	/* == vmag cos( true/2 + Pi/2 ) */ 
      vv.y =   vmag * cos( 0.5*trueAnomaly );	/* == vmag sin(  etc. ) */ 
      vv.z = 0.;
  }
  else
  {
      printf(" in navVelocity, orbit not a valid conic section (?!), ecc = %e \n", 
	     orbit->o_eccen );
      abort();
  }


  matEulerToMat(&orbit->o_er,mat);
  /* euler angles in orbital elements are world to orbit; we need reverse: */
  /* matTranspose(mat,matx); */
  matMatVV( mat, &vv, velocity );
  
}


/* ----------------------------------------------------------------
 * navPosVelToOrbit
 *	orbit from starting position and velocity
 * inputs
 *	mass of central body
 *	time, double float, in seconds
 * 	cartesian position 
 * 	cartesian velocity
 *	pointer to orbit, where to put the results
 * outputs	
 * 	none
 * side effects
 * 	sets elements of orbit
 * sources
 *	BMW, p. 62 ff., (sometimes).
 * ----------------------------------------------------------------
 */

void
navPosVelToOrbit( mass, time, pos, vel, orb)
     double  mass;	/* mass of central body */
     double time; 	/* time at which pos and vel are specified */ 
     Coord3d *pos;	/* geocentric position */
     Coord3d *vel;	/* velocity in same coord system */
     Orbit *orb;	/* if not null on input, this will be used */ 
{
    Coord3d upos;	/* unit vector in direction of position */
    Coord3d direction;	/* unit vector in direction of velocity */
    Coord3d pole;	/* unit vector perpendicular to the orbit */
    Coord3d nodevector;	/* unit vector in direction of the line of nodes */
    Coord3d solstice;	/* unit vector 90 degrees counter-clockwise 
			   from nodevector in orbit plane */ 
    Coord3d eccVec;	/* eccentricity vector (BMW, p. 62 or p. 25-26) */ 
    Coord3d eccUnit;	/* unit vector in direction of eccentricity if e != 0; 
			   else 0-vector */ 
    double   eccentricity;  /* eccentricity */
    Coord3d angMomVec;	/* angular momentum */
    double   angMom2;	/* scalar angular momentum squared */
    double   energy;	/* total energy */ 

    Coord3d zunit;	/* z unit vector */
    Coord3d latusVector; /* unit vector to the latus rectum:
			    pole cross eccentricity vector = latus vector */  
    double   semiLatusRectum;	/* semi-latus-rectum */ 
    double   node;	/* longitude of line of nodes */
    double   incl;	/* orbital inclination */
    double   pgee;	/* argument of the perigee */
    double   axis;	/* semimajor axis */ 	
    double   trueAnomaly; /* the true anomaly */
    double   eccAnomaly;	/* eccentric anomaly */
    double   meanAnomaly; /* mean anomaly */
    double   vel2;	/* velocity squared */
    double   mu;		/* GM */
    double   radius;	/* radius at this position */
    Coord3d v1;		/* dummy vector */
    Coord3d v2;		/* dummy vector */
    double   xx;		/* dummy scalar */
    double   pgx, pgy;	/* dummies on way to get pgee */



    mu = BIG_G * mass;
    vel2 = GeoDotProduct( vel, vel );
    radius = GeoVectorLength( pos );
    GeoCrossProduct( pos, vel, &angMomVec); 
    angMom2 = GeoDotProduct( &angMomVec, &angMomVec);

    energy = vel2/2. - mu/radius;
    /* energy < 0. --> ellipse, == 0. --> parabola, > 0. --> hyperbola */ 

    GeoUnitVector( pos, &upos);
    GeoUnitVector( vel, &direction );


    GeoCrossProduct( &upos, &direction, &pole);
    GeoUnitVector( &pole, &pole);
    incl = acos(pole.z);

    /* vectors to the line of nodes and to the solstice */
    zunit.x = 0.; zunit.y = 0.; zunit.z = 1.;
    GeoCrossProduct( &zunit, &pole, &nodevector);
    if (GeoVectorLength ( &nodevector) < 1.e-6)
    {	/* inclination is nearly 0 or 180 */
	nodevector.x = 1.;
	nodevector.y = 0.;
	nodevector.z = 0.;
    }
    GeoUnitVector( &nodevector, &nodevector);

    GeoCrossProduct( &pole, &nodevector, &solstice );
    GeoUnitVector( &solstice, &solstice );
    node = atan2( nodevector.y, nodevector.x );

    /* eccentricity, BMW, p. 62, eqn (2.4-5):  */
    xx = (vel2 / mu - 1. / radius);
    GeoScalarMultVector( xx, pos, &v1);
    xx = - GeoDotProduct( pos, vel) / mu;
    GeoScalarMultVector( xx, vel, &v2);
    GeoAddVector( &v1, &v2, &eccVec);
    eccentricity = GeoVectorLength( &eccVec);

    /* alternate and equivalent method */ 
    /* BMMA, p. 115, eq (3.14) */ 
    GeoCrossProduct( pos, vel, &v1 );	/* v1 == ang-mom, RxV */
    GeoCrossProduct( vel, &v1, &v2 ); 	/* v2 == Vxh, V x ang-mom */
    GeoUnitVector( pos, &direction ); 
    GeoScalarMultVector( mu, &direction, &direction ); 
    GeoSubtractVector( &v2, &direction, &eccVec );
    GeoScalarMultVector( 1./mu, &eccVec, &eccVec ); 


    GeoUnitVector( &eccVec, &eccUnit);

    if ( ABS(energy) < vel2 * 1.e-7 )	/* a parabola */ 
    {
	if (ABS(eccentricity -1.) > 1.e-8)
	  printf(" suspicious eccentricity = %e in near parabolic orbit, energy = %e \n",
		 eccentricity, energy ); 
    }
    if (eccentricity == 0.0 )
    {
      GeoScalarMultVector( 1., &nodevector, &eccUnit);
    }

    GeoCrossProduct( &pole, &eccUnit, &latusVector);
    GeoUnitVector( &latusVector, &latusVector ); 

    /* BMW, p. 24, eq. (1.5-6) and p. 26, eq. (1.6-1): */
    semiLatusRectum = angMom2 / mu; 		/* this ALWAYS exists */ 
    if (eccentricity < 1.0-PARACRIT  ||  eccentricity > 1.0+PARACRIT )
      axis = semiLatusRectum /(1. - eccentricity * eccentricity );
    else
    {
	axis = semiLatusRectum; 	/* this case is different, yrrrch !! */ 
    }

    /* resolving quadrant ambiguity in the perigee is a little trickier */
    pgy = GeoDotProduct( &eccUnit, &solstice);
    pgx = GeoDotProduct( &eccUnit, &nodevector);
    pgee = atan2( pgy, pgx);
/*
    printf(" latus vector %e %e %e \n", latusVector.x, latusVector.y, latusVector.z);
    printf(" ecce  vector %e %e %e \n", eccUnit.x, eccUnit.y, eccUnit.z);
    printf(" upos  vector %e %e %e \n", upos.x, upos.y, upos.z);
*/    
    trueAnomaly = atan2( 
			GeoDotProduct( &latusVector, &upos), 
			GeoDotProduct( &eccUnit, &upos) 
			);
    if ( ABS( eccentricity - 1. ) > PARACRIT) 	/* not a parabola */ 
    {
	eccAnomaly =  navTrueToEccentricAnomaly( eccentricity, trueAnomaly);
	meanAnomaly = navEccToMeanAnomaly(       eccentricity, eccAnomaly);
    }
    else	/* a parabola */ 
    {
	meanAnomaly = parabolicTrueToMeanAnomaly( trueAnomaly ); 
    }


    /* collect the elements: */
    orb->o_mass		= mass;
    orb->o_eccen	= eccentricity; 
    orb->o_axis		= axis;

    orb->o_er.er_1	= node;
    orb->o_er.er_2	= incl;
    orb->o_er.er_3	= pgee;

    orb->o_refTime	= time; 	/* thus refTime = now   */
    orb->o_refAnomaly	= meanAnomaly;

    navResetPeriTime( orb); 

}


/* ----------------------------------------------------------------
 * navLocalVertical
 * 	finds unit vectors in LVS,
 *	given orbit and time for which LVS is sought
 * method
 *	create unit matrix; at perigee, 
 *		top row = x^ = -down
 * inputs
 *	orbit
 *	time
 *	pointers to 3 vectors 
 * outputs
 *	none
 * side effects
 *	sets *up, *forward, *left:
 * method
 *	*up || position, from navAnomalyToPosition
 *	*forward || velocity, from navVelocity
 *	*left (cross) *up = *forward 
 * ----------------------------------------------------------------
 */

void
navLocalVertical(orbit,trueAnomaly,up,forward,left)
  Orbit *orbit;
  double trueAnomaly;
  Coord3d *up;
  Coord3d *forward;
  Coord3d *left;
{
  Coord3d position, velocity, perpendicular;

  navAnomalyToPosition( orbit, trueAnomaly, &position );
  navVelocity( orbit, trueAnomaly, &velocity );

  GeoUnitVector(&position, up);
  GeoCrossProduct(up, &velocity, &perpendicular);
  GeoUnitVector(&perpendicular, left);
  GeoCrossProduct(left, up, forward);

}



/* ----------------------------------------------------------------
 * navMeanMotion calculates mean motion, radians per second
 * input
 *	orbit
 * output 
 *	mean motion
 * side effects
 *	none
 * ----------------------------------------------------------------
 */

double
navMeanMotion( orb )
  Orbit *orb;
{
  double mu;
  double axis;

  mu = BIG_G * orb->o_mass;
  if ( !parabola( orb ) ) 
  {
      axis = ABS( orb->o_axis );		/* (its negative for hyperbolas) */ 
      return sqrt( mu / (axis * axis * axis ));
  }
  else 
  {
      return 0.; 	
      /* this is strictly true for a parabola -- but useless;  
       * the relevant rate is given by 
       *             4. * sqrt( mu / (p * p * p )) 
       */ 
  }


}
    

/* ---------------------------------------------------------------- 
 * parabolicRate gets a rate that is used in Barker's equation and other places,
 * and which gives the rate of change for the parabolic surrogate for a mean anomaly
 *	(mean motion for a parabola is zero -- exactly)
 * this routine parallels navMeanMotion() 
 * ----------------------------------------------------------------
 */ 



double  
parabolicRate( orb ) 
  Orbit *orb;
{
  double mu;
  double p; 

  if ( !parabola( orb ) ) 
  {
      printf(" you called parabolicRate for a non-parabola \n");
  }
  /* the relevant rate is given by (here, axis = semi-parameter):  */ 
  mu = BIG_G * orb->o_mass;
  p = orb->o_axis;
  return        4. * sqrt( mu / (p * p * p )); 

}
    






/* ----------------------------------------------------------------
 * navRefAnomaly calculates reference anomaly
 * input
 *	orbit
 * output 
 *	reference anomaly
 * side effects
 *	none
 * method
 *	it assumes that reference time o_refTime is present in *orb, 
 *	and that o_periTime and o_er.er_3 are also present
 *	calls navMeanAnomaly()
 * ----------------------------------------------------------------
 */

double
navRefAnomaly( orb )
  Orbit *orb;
{
    double xx;
    xx = navMeanAnomaly( orb, orb->o_refTime );
    return xx;
}

/* ----------------------------------------------------------------
 * navResetPeriTime() makes perigee time o_periTime 
 *	consistent with o_refTime and o_refAnomaly
 * inputs
 *	orbit
 * outputs
 *	none
 * side effects
 * 	sets o_periTime
 * ----------------------------------------------------------------
 */

void
navResetPeriTime( orb )
     Orbit *orb;
{
    double rate; /* meanMotion, for ellipse/hyperbola; quite different for parabola */ 

    double deltaAnomaly;
    double deltaTime;

    if ( !parabola( orb ) )
    {
	rate  = navMeanMotion( orb ) ;
    }
    else	/* parabola is a special case; its mean motion is zero */ 
    {
	rate = parabolicRate( orb ); 
    }


    deltaAnomaly = orb->o_refAnomaly;
    deltaTime  = (double) ( deltaAnomaly / rate ); 

    orb->o_periTime = orb->o_refTime - deltaTime; 

}



/* ----------------------------------------------------------------
 * to shift the ref time and ref-anomaly: 
 * ----------------------------------------------------------------
 */

void
shiftRefTime( orb, time )
     Orbit *orb;	/* the orbit to be shifted */ 
     double time; 	/* new ref-time */ 
{
    double rate; 

    if (  !parabola( orb ) )
    {
	rate = navMeanMotion( orb ); 
    }
    else
    {
	rate = parabolicRate( orb );
    }

    orb->o_refAnomaly += rate * ( time - orb->o_refTime ); 
    orb->o_refTime = time; 

}







/* ----------------------------------------------------------------
 * parabola() returns TRUE if orbit is a parabola, FALSE otherwise 
 * ellipse()
 * hyperbola() similarly . . . 
 *	it merely inspects the value of the eccentricity
 *	if within tolerance (PARACRIT) of 1.0, it's a parabola
 * similarly for hyperbola() and ellipse() . . . 
 * ----------------------------------------------------------------
 */

int 
parabola( orb )
     Orbit *orb;	/* the orbit to be inspected */ 
{
    if ( (orb->o_eccen < (1.0 - PARACRIT) )  ||  ( orb->o_eccen > (1.0 + PARACRIT)  ))
      return FALSE;
    else
      return TRUE;
}	

int
hyperbola( orb )
     Orbit *orb;	/* the orbit to be inspected */ 
{
    if ( orb->o_eccen > (1.0 + PARACRIT) )
      return TRUE; 
    if ( orb->o_eccen < ( -1.0 - PARACRIT) )
      return TRUE; 
    else
      return FALSE; 
}	

int 
ellipse( orb )      
     Orbit *orb;	/* the orbit to be inspected */ 
{
    if ( orb->o_eccen < (1.0 - PARACRIT) )
      return TRUE; 
    else
      return FALSE; 
}	


/* ----------------------------------------------------------------
 * barkersEquation returns the solution of Barker's equation,
 *	or other method if that works poorly
 * inputs 
 *	orbit
 *	time
 * output
 *	true anomaly 
 * side effects 
 * 	none
 * ----------------------------------------------------------------
 */ 

double 
barkersEquation( orb, time )
     Orbit *orb;
     double time;
{
    double B, A;		/* as used at BMMA, pp. 150-151 */ 
    double p;			/* semi-parameter */ 
    double tanhalftheta;	/* tan( 1/2 true anomaly ) */ 
    double radical;		/* cuberoot( B + sqrt(1 + B*B) ) */ 
    double trueAnomaly;
    double mu; 
    
    if ( !parabola( orb ) ) 
    {
	printf(" you called barkersEquation for a non-parabolic orbit! \n" );
	return 0.;
    }

    p = orb->o_axis;	/* note special case for parabola */ 
    mu = orb->o_mass * BIG_G; 

    B = 3. * sqrt( mu/ (p*p*p) )  *  (time - orb->o_periTime); 	/* eq (4.10 */ 
    radical = B + sqrt( 1. + B*B ); 
    radical = exp( log( radical/3. ) );		/* cube root */ 
    
    if ( ABS(radical - 1.) > 1.e-3 )
    {
	tanhalftheta = radical - 1./radical;		/* eq (4.12) */ 
    }
    else 	/* too close to periapse to get accurate answer from eq (4.12) */ 
    {
	A = radical * radical;
	tanhalftheta = 2.*A*B / (1. + A + A*A); 	/* eq (4.17) */ 
    }

    trueAnomaly = 2. * atan( tanhalftheta ); 
    return trueAnomaly; 

}


/* ----------------------------------------------------------------
 * parabolicTrueToMeanAnomaly is the inverse of Barker's Equation 
 * inputs
 *	true anomaly
 * outputs
 * 	mean anomaly
 * side effects 
 *	none
 * ----------------------------------------------------------------
 */

double 
parabolicTrueToMeanAnomaly( trueAnomaly )
     double trueAnomaly;
{
    double tanhalftrue;		/* tan( trueAnomaly/2. )  */
    double meanAnomaly;

    tanhalftrue = tan( trueAnomaly * 0.5 ); 

    meanAnomaly = 2. * ( tanhalftrue*tanhalftrue*tanhalftrue/3. + tanhalftrue );

    return meanAnomaly;

}



/* ----------------------------------------------------------------
 * this is so that a bad arg to sqrt() or acos() 
 *	will cause a crash/abort, allowing dbx,
 *	rather than running amok calculating with NaN's
 * see C-manual; you can omit this routine altogether
 *	if you don't want to crash on bad args to math funcs 
 * ----------------------------------------------------------------
 */ 


int
matherr( exc )
     struct exception *exc;
{
    /* we just crash . . . */ 

    double r; 
    double x[3]; 

    printf(" about to crash for dbx, in matherr \n" ); 

    /* cast not necessary Sparc; gcc on Linux complains without it: */ 
    fflush( (FILE *) 1 );
    fflush( (FILE *) 2 ); 

    abort(); 

}


