/* 
 * orbUtility.c utility subroutines for orbit finder 
 * 88/07/14 ff.;
 * some of these are "small" routines with simple operatons;
 * but the rotate-orbit routines are longer and more complex. 
 * they were originally in celcoord.c, then in rotorbit.c, 
 * but they seem to belong naturally here as general orbit utilities 
 */


#include <stdio.h>
#include <math.h>

#include "../include/ms.h"
#include "../include/matrix.h"
#include "../include/hnav.h"



extern char *malloc();



 
/* ----------------------------------------------------------------
 * copyOrbit copies elements from its first *argument to its *second
 * ----------------------------------------------------------------
 */

void
copyOrbit(oy,ox)
     Orbit *oy;
     Orbit *ox;
{
  ox->o_mass		= oy->o_mass;
  ox->o_eccen 		= oy->o_eccen;
  ox->o_axis 		= oy->o_axis;
  ox->o_periTime 	= oy->o_periTime;
  ox->o_er.er_1 	= oy->o_er.er_1;
  ox->o_er.er_2 	= oy->o_er.er_2;
  ox->o_er.er_3 	= oy->o_er.er_3;
  ox->o_refAnomaly 	= oy->o_refAnomaly;
  ox->o_refTime 	= oy->o_refTime;
}






/* ----------------------------------------------------------------
 * howBad calculates how bad the positions calculated from its second orbit are,
 *	measured from its first orbit
 * inputs
 *	ptrs to two orbits
 * outputs
 *	maximum position error encountered, in METERS 
 * side effects
 *	prints std dev and max position error
 * method
 *	calculates 100 positions around both orbits, and compares them
 * ----------------------------------------------------------------
 */

double 
howBad(orbx,orby)
     Orbit *orbx;
     Orbit *orby;
{
    Coord3d px, py;	/* positions in orbits x and y */
    Coord3d dp;		/* difference of same */
    double   dr;		/* scalar length of same */
    int i;		/* index */
    int npts;		/* number of points */
    double maxError;	/* max abs position error, in METERS */
    double stddev;	/* std dev position error */	
    double sumdev;	/* used to calculate stddev */
    double period;	/* orbital period */
    double dt;		/* time step */
    double time;		/* measured from o_periTime */ 

    if (OKOrb(orbx) != 1) 
    {
	printf("trubble: \n");
	printf("bad orbx in howBad() \n");
	dumpOrbit(orbx); 
	abort();
    }
    if (OKOrb(orby) != 1) 
    {
	printf("trubble: \n");
	printf("bad orby in howBad() \n");
	dumpOrbit(orby); 
	abort();
    }

    period = 2.*PI / navMeanMotion(orbx);
    npts = 100;
    dt = period / npts;

    maxError = 0.;
    i = 0; 
    for ( time=orbx->o_periTime;  i<npts;   ++i, time += dt)
    {
	navKepler( orbx, time, &px );
	navKepler( orby, time, &py );
	GeoSubtractVector( &px, &py, &dp );
	dr = GeoVectorLength( &dp );

	maxError = MAX( maxError, dr);
    }

    return maxError;
    
}





/* ----------------------------------------------------------------
 * diffOrbit() prints difference between two orbits 
 * inputs
 *	two orbits
 * outputs 
 * 	none
 * side effects
 * 	lists differences in orbital elements
 * ----------------------------------------------------------------
 */

void
diffOrbit(orbx,orby)
     Orbit *orbx;
     Orbit *orby;
{
    Orbit orbd;

    orbd.o_eccen 	= orbx->o_eccen - orby->o_eccen;
    orbd.o_periTime 	= orbx->o_periTime - orby->o_periTime;
    orbd.o_axis		= orbx->o_axis - orby->o_axis;
    orbd.o_er.er_1	= orbx->o_er.er_1 - orby->o_er.er_1;
    orbd.o_er.er_2	= orbx->o_er.er_2 - orby->o_er.er_2;
    orbd.o_er.er_3	= orbx->o_er.er_3 - orby->o_er.er_3;
    orbd.o_refAnomaly	= orbx->o_refAnomaly - orby->o_refAnomaly;
    orbd.o_refTime	= orbx->o_refTime - orby->o_refTime;

    dumpOrbit(&orbd);

}


/* ----------------------------------------------------------------
 * equalOrbits returns 1 for two orbits equal, 0 for unequal
 * inputs
 * 	ptrs to two orbits
 * outputs
 *	1 for equal, 0 for unequal 
 * side effects
 * 	none
 * ----------------------------------------------------------------
 */ 

int equalOrbits( orb1, orb2 )
     Orbit *orb1;
     Orbit *orb2;
{

    if ( orb1->o_mass != orb2->o_mass ) 		return 0; 
    if ( orb1->o_periTime != orb2->o_periTime ) 	return 0; 
    if ( orb1->o_eccen != orb2->o_eccen )		return 0; 
    if ( orb1->o_axis != orb2->o_axis )			return 0;
    if ( orb1->o_er.er_1 != orb2->o_er.er_1 )		return 0;
    if ( orb1->o_er.er_2 != orb2->o_er.er_2 )		return 0;
    if ( orb1->o_er.er_3 != orb2->o_er.er_3 )		return 0;
    if ( orb1->o_refAnomaly != orb2->o_refAnomaly )	return 0;
    if ( orb1->o_refTime != orb2->o_refTime )		return 0;

    return 1;

}


/* ----------------------------------------------------------------
 * OKOrb returns 1 for ok orbital elements, 0 for not-ok (NaN, etc.) 
 * these are rather lenient conditions for non-absurdity, 
 * but they should detect NaN and over/under-flow
 * ----------------------------------------------------------------
 */

int 
OKOrb(orb)
     Orbit *orb;
{
    double xx;
    int fail;	 /* fail == 1 means some element is bad */
    fail = 0;

    if (!(orb->o_mass > 1.e0)) 
      {
	  printf(" in OKOrbit, orbit mass is not plausible \n");
	  fail = 1; 
      }
    if (!(orb->o_mass < 1.e50)) 	/* a million galaxies ?? */ 
      {
	  printf(" in OKOrbit, orbit mass is not plausible \n");
	  fail = 1;
      }
    if (orb->o_eccen < 0. ) 		
      {
	  printf(" in OKOrbit, orbit eccentricity is not plausible \n");
	  fail = 1; 
      }
    xx = orb->o_periTime;
    if (!(xx > -3.15e17 && xx < 3.15e17))   /* age of the universe */ 
      {
	  printf(" in OKOrbit, orbit perigee time is not plausible \n");
	  fail = 1; 
      }

    if ( orb->o_axis  < 10. && orb->o_eccen < 1.+PARACRIT ) 	
      {
	  printf(" in OKOrbit, orbit axis is not plausible "); 
	  printf(" for an elliptical or parabolic orbit \n");
	  fail = 1; 
      }
    if ( orb->o_axis  > -10. && orb->o_eccen > 1.+PARACRIT ) 
      {
	  printf(" in OKOrbit, orbit axis not plausible for hyperbola  \n");
	  fail = 1; 
      }



    if (!(orb->o_er.er_1 >= -6.3 && orb->o_er.er_1 <= 6.3))
      {
	  printf(" in OKOrbit, orbit node is not reduced \n");
	  fail = 1; 
      }
    if (!(orb->o_er.er_2 >= -6.3 && orb->o_er.er_2 <= 6.3)) 
      {
	  printf(" in OKOrbit, orbit inclination is not reduced \n");
	  fail = 1; 
      }
    if (!(orb->o_er.er_3 >= -6.3 && orb->o_er.er_3 <= 6.3)) 
      {
	  printf(" in OKOrbit, orbit perigee is not reduced \n");
	  fail = 1; 
      }

    return !fail;
}



/* rotorbit.c: routines taken from celcoord.c 94/06/07,
 * moved to orbutili.c 94/09/27 
 */ 

/* ----------------------------------------------------------------
 * testRotateOrbit tests rotateOrbit 
 * ----------------------------------------------------------------
 */ 

void
testRotateOrbit( )
{
     Orbit orba;	/* orbit to rotate */ 
     Orbit orbb;	/* two rotated versions of it  */ 
     Orbit orbc;
     matrix3d mat;	/* the coordinate rotation matrix */
     matrix3d tempmat;	/* temp storage */ 

     double node, incl, peri; 	/* euler angles of the orbit */ 
     double phi, theta, psi;    /* euler angles of the rotation */ 


     orba.o_mass = 1.;
     orba.o_axis = 1000.;
     orba.o_eccen = 0.2;
     orba.o_periTime = 0.;
     orba.o_refAnomaly = 0.;
     orba.o_refTime = 0.;

     for ( peri=0.0; peri<0.2; peri+=0.1 )
       for ( incl=0.0; incl<0.2; incl+=0.1 )
	 for ( node=0.0; node<0.2; node+=0.1 )
	 {
	     printf("\n orbit node  %f incl  %f peri  %f \n", node, incl, peri ); 
	     orba.o_er.er_1 = node;
	     orba.o_er.er_2 = incl;
	     orba.o_er.er_3 = peri;

	     psi = 0.;
	     theta = 0.0; 
	     for ( psi=0.; psi>-0.2; psi-=0.05 )
	       for ( theta=0.; theta>-0.2; theta-=0.05 ) 
		 for ( phi=0.; phi>-0.2; phi-=0.05 )
		 {
		     copyOrbit( &orba, &orbb );
		     copyOrbit( &orba, &orbc ); 

		     printf("\n trans phi   %f theta %f psi   %f \n", phi, theta, psi ); 
		     matERot( phi, theta, psi, mat ); 
		     rotateOrbit( &orbb, mat ); 

		     matEulerToMat( &(orbb.o_er), tempmat );

		     rotateOrbitVectors( &orbc, mat ); 

		     matEulerToMat( &(orbc.o_er), tempmat );
		     printf(" orba "); dumpOrbit( &orba ); 
		     printf(" orbb "); dumpOrbit( &orbb );
		     printf(" orbc "); dumpOrbit( &orbc ); 
		     printf(" diff "); diffOrbit( &orbb, &orbc ); 

		     
		 }
	 }
}



/* ----------------------------------------------------------------
 * rotateOrbit changes the euler angles 
 *	as per rotation matrix supplied 
 * inputs 
 * 	pointer to an orbit 
 * 	rotation matrix 
 * outputs
 * 	none 
 * side effects 
 * 	changes the euler angles 
 * compare rotateOrbitVectors(); it should get the same answers 
 * ----------------------------------------------------------------
 */ 

void
rotateOrbit( orb, mat )
     Orbit *orb;	/* orbit to rotate */ 
     matrix3d mat;	/* the coordinate rotation matrix */
{
    matrix3d emata;	/* the rotation matrix of the euler angles */ 
    matrix3d ematb;	/* the transformed matrix of euler angles */ 
    matrix3d temp;
    matrix3d transmat;	/* transpose of mat */ 

    /* emata is orbit-to-old-ref system
            i. e., emata x (1,0,0) gives the eccentricity unit-vector
       mat is old-ref to new-ref system
       so [ mat  x  emata ] is orbit-to-new-ref system trans matrix
       from which may then be recovered the new euler angles: 
     */ 


    matEulerToMat( &(orb->o_er), emata ); 

    matMult( mat, emata, ematb );
      
    matMatToEuler( ematb, &(orb->o_er) ); 

}


/* ----------------------------------------------------------------
 * rotateOrbitVectors changes the euler angles 
 *	as implied by the supplied rotation matrix  
 *	from one coord system to another 
 * inputs 
 * 	pointer to an orbit 
 * 	rotation matrix 
 * outputs
 * 	none 
 * side effects 
 * 	changes the euler angles 
 * compare rotateOrbit 
 *	this routine calculates vectors defining the orbit, 
 *	and rotates them, and recovers euler angles;
 *	that routine does it all by matrix operations 
 * ----------------------------------------------------------------
 */ 

void
rotateOrbitVectors( orb, mat )
     Orbit *orb;	/* orbit to rotate */ 
     matrix3d mat;	/* the coordinate rotation matrix */
{
    Coord3d h;		/* the angular momentum unit vector */
    Coord3d nodevector; /* the ascending node unit vector */
    Coord3d evector; 	/* the eccentricity unit vector */ 
    double cincl, sincl, cnode, snode, cpgee, spgee;  /* trig funcs of euler angles */ 
    double node, incl, pgee; 	/* euler angles */ 
    Coord3d tempvec;	/* temp storage */ 
    double tempx;	/* temp for scalars */ 

    /* the method: 
      rotation matrix is given to us, we don't need to get it.  
      copy euler angles, get their trig functions 
      get h, omega, eccen in from-system 
      rotate h, omega, eccen by the given rotation matrix to the to-system 
      recover euler angles from h, omega, eccen
      alter euler angles in the input orbit 
      correct the reference anomaly and reference time 
      */ 

    node = orb->o_er.er_1;
    incl = orb->o_er.er_2;
    pgee = orb->o_er.er_3;

    sincl = sin(incl); cincl = cos(incl);
    snode = sin(node); cnode = cos(node);
    spgee = sin(pgee); cpgee = cos(pgee); 

    /* the angular momentum vector, in the <from> coordinate system: */ 
    h.x = sincl * snode;	/* <-90 degrees from line of nodes */
    h.y = - sincl * cnode;	/* in xy plane> == proj of h-vector */  
    h.z = cincl; 		/* h projected on old z-axis */ 

    /* the node vector is relatively simple */ 
    nodevector.x = cnode;
    nodevector.y = snode;
    nodevector.z = 0.0; 

    /* the eccentricity unit vector is harder; 
       * evector = cos(pgee) * nodevector + sin(pgee) * ( h X nodevector )
       * ( h X nodevector ) = ( -cincl*snode, cincl*cnode, sincl )
       * i. e., z-component = sin(incl); clear enough 
       * xy proj of h X nodevector is cos(incl) times nodevector rotated +90 degrees
       * 
       * summing to get evector gives 
       * evector = 
       *	( cpgee * cnode - spgee * cincl * snode, 
       *	  cpgee * snode + spgee * cincl * cnode, 
       *	                  spgee * sincl )
       */ 
    evector.x = cpgee * cnode - spgee * cincl * snode; 
    evector.y = cpgee * snode + spgee * cincl * cnode;  
    evector.z =                 spgee * sincl; 

    /* now rotate the vectors that are peculiar to the orbit alone: */ 
    /* (note, in matrix.c, that invec=outvec is permitted) */ 
    matMatVV( mat, &h, &h );
    matMatVV( mat, &evector, &evector );

    /* the node vector has to be determined from the orbit _and_ new coord system: */ 
    /* in fact, new node can be found from new ang-mom vector alone: */ 
    nodevector.z = 0.0; 	/*   ( by definition! )   */ 
    if ( h.z < 1.0   &&   h.z > -1.0 )
    {
	tempx = 1.0 / sqrt( 1.0 - h.z*h.z ); 
	nodevector.x = - h.y * tempx;  
	nodevector.y =   h.x * tempx;  
    }
    else
    {
	nodevector.x = 1.0;
	nodevector.y = 0.0;
    }


    /* now recover the euler angles: */ 
    incl = acos( h.z ); 
    node = atan2( nodevector.y, nodevector.x ); 
    node = normalize( node, 0., PI2 ); 
    /* tempvec = h X nodevector: */
    GeoCrossProduct( &h, &nodevector, &tempvec );
    pgee = atan2( 
		 GeoDotProduct( &evector, &tempvec ), 
		 GeoDotProduct( &evector, &nodevector ) 
		 ); 
    pgee = normalize( pgee, 0., PI2 ); 

    /* now save the euler angles; you have rotated the orbit onto the to-system: */ 
    orb->o_er.er_1 = node; 
    orb->o_er.er_2 = incl; 
    orb->o_er.er_3 = pgee; 

    /* you are done:  you do not have to correct the reference anomaly; 
       * (the reference anomaly is defined with reference to a new perigee angle)
       * BUT -- we assume that the input orbit had a valid periapse time, 
       * for the time of periapse passage is invariant with respect to coord-sys changes 
       * likewise, the reference time is arbitrary, but supplied with the orbit:
       * and the mean anomaly is a function only of these two times, 
       * which do not change 
       */

}
