/* ----------------------------------------------------------------
 * precnuta.c: mostly precession and nutation routines
 * ----------------------------------------------------------------
 */

/* ----------------------------------------------------------------
 * files from pnsw.c 
 * 	this is a translated and altered c version of fortran routines
 *	from novas.f, 
 * from usno: us naval observatory apparent place software
 *	1989/05/11 ff.; by APP 
 *	not all the novas.f routines are here . . . 
 * only
 *	PRECESSION and NUTATION
 * 	. . . and the 106-term series evaluation for nutation is not here;
 *	it is in nod.f, to be compiled in Fortran and linked by the C compiler
 * ----------------------------------------------------------------
 */

#include <stdio.h>
#include <math.h>

#include "../include/ms.h"
#include "../include/matrix.h"
#include "../include/hnav.h"

void etilt();


/* ----------------------------------------------------------------
 *
 *      subroutine preces (tjd1,pos1,tjd2,pos2)                                   
 * 	these arguments have bean CHANGED !!!! below
c                                                                               
c     this subroutine precesses equatorial rectangular coordinates from         
c     one epoch to another.  the coordinates are referred to the mean           
c     equator and equinox of the two respective epochs.  see pages 30-34        
c     of the explanatory supplement to the ae, lieske, et al. (1977)            
c     astronomy and astrophysics 58, 1-16, and lieske (1979) astronomy          
c     and astrophysics 73, 282-284.                                             
c                                                                               
c          tjd1 = tdb julian date of first epoch (in)                           
c          pos1 = position vector, geocentric equatorial rectangular            
c                 coordinates, referred to mean equator and equinox of          
c                 first epoch (in)                                              
c          tjd2 = tdb julian date of second epoch (in)                          
c          pos2 = position vector, geocentric equatorial rectangular            
c                 coordinates, referred to mean equator and equinox of          
c                 second epoch (out)                                            
c                                                                               
c                                                                               
 * we do not rotate a vector, as the USNO Fortran subroutine did,
 *	but simply set the precession parameters zeta, z, theta
 * inputs
 *	tjd1, tjd2: the epochs from and to which we precess
 *	ptrs to zeta, z, theta
 * outputs
 *	none
 * side effects
 *	sets zeta, z, theta
 * ----------------------------------------------------------------
 */

void
usnoPrecess( tjd1, tjd2, zeta, z, Theta )
     double tjd1;		/* JD of first epoch */
     double tjd2;		/* JD of second epoch */
     double *zeta;		/* zeta, z, theta are constants defined */
     double *z;			/* in the theory of precession          */
     double *Theta;

{

      double t0,t,t02,t2,t3;
      double zeta0,zee,theta,czeta0,szeta0,czee,szee,ctheta,stheta;               


      double seccon;

      seccon=206264.8062470964;                                          

                                                                               
      /*      t0 and t below correspond to lieske's big t and little t     */
      t0 = (tjd1 - 2451545.0) / 36525.0;                                     
      t = (tjd2 - tjd1) / 36525.0;                                             
      t02 = t0 * t0;                                                             
      t2 = t * t;                                                                
      t3 = t2 * t;                                                               
      /*     zeta0, zee, and theta below correspond to lieske's zeta-sub-a, */
      /*     z-sub-a, and theta-sub-a                                       */
      zeta0 = (2306.2181 + 1.39656*t0 - 0.000139*t02) * t                 
           + (0.30188 - 0.000344*t0) * t2                                  
           +  0.017998 * t3;                                                  
      zee   = (2306.2181 + 1.39656*t0 - 0.000139*t02) * t                 
           + (1.09468 + 0.000066*t0) * t2                                  
           +  0.018203 * t3;                                                  
      theta = (2004.3109 - 0.85330*t0 - 0.000217*t02) * t                 
           + (-0.42665 - 0.000217*t0) * t2                                 
           -  0.041833 * t3;
                                                  
      zeta0 = zeta0 / seccon;                                                    
      zee = zee / seccon;                                                        
      theta = theta / seccon;                                                    

      *zeta = zeta0;
      *z = zee;
      *Theta = theta;

  }




/* ----------------------------------------------------------------
 *      subroutine nutate (tjd,pos1,pos2)                                         
c                                                                               
c     this subroutine nutates equatorial rectangular coordinates from           
c     mean equator and equinox of epoch to true equator and equinox of          
c     epoch.  see pages 41-45 of the explanatory supplement to the ae.          
c                                                                               
c          tjd    = tdb julian date of epoch (in)                               
c          pos1   = position vector, geocentric equatorial rectangular          
c                   coordinates, referred to mean equator and equinox           
c                   of epoch (in)                                               
c          pos2   = position vector, geocentric equatorial rectangular          
c                   coordinates, referred to true equator and equinox           
c                   of epoch (out)                                              
c                                                                               
c          note --  if tjd is negative, inverse nutation (true to mean)         
c                   is applied                                                  
c                                                                               
c                                                                               
 * we have made a MAJOR ALTERATION in the c-version: 
 *	we do not calculate a rotated vector
 *	we only set the nutation matrix, pointed to by matrix3d mat
 * 
 * ----------------------------------------------------------------
 */

void
usnoNutate( tjd, mat )
     double tjd;		/* Julian Day for which nutation is desired */
     matrix3d mat;		/* rotation matrix for nutation */
{

      double tjd1;		/* time input, julian day */
      double seccon;		/* seconds per radian; conversion factor */
      double oblm,oblt;		/* mean and true obliquity */
      double eqeq; 		/* equation of equinoxes in seconds of time */
      double dpsi,deps;		/* nutations is longitude and obliquity */

      double cobm,sobm,cobt,sobt,cpsi,spsi;	/* trig functions */
      double xx,yx,zx,xy,yy,zy,xz,yz,zz;

      seccon = 206264.8062470964;                                          

                                                                               
      /* tjd1 = dabs(tjd) this is strange fortran; negative julian date ??!! */
      tjd1 = tjd;
                                                                               
      etilt ( tjd1, &oblm, &oblt, &eqeq, &dpsi, &deps ); 

      oblm = oblm * 3600.0 / seccon;                                           
      oblt = oblt * 3600.0 / seccon;                                           
      dpsi = dpsi / seccon;                                                      
      deps = deps / seccon;                                                      
      cobm = cos(oblm);                                                         
      sobm = sin(oblm);                                                         
      cobt = cos(oblt);                                                         
      sobt = sin(oblt);                                                         
      cpsi = cos(dpsi);                                                         
      spsi = sin(dpsi);                                                         
                                                                               
/*      nutation rotation matrix follows   */
      xx = cpsi;
      yx = -spsi*cobm;                                                           
      zx = -spsi*sobm;                                                           
      xy = spsi*cobt;                                                            
      yy = cpsi*cobm*cobt + sobm*sobt;                                           
      zy = cpsi*sobm*cobt - cobm*sobt;                                           
      xz = spsi*sobt;                                                            
      yz = cpsi*cobm*sobt - sobm*cobt;                                           
      zz = cpsi*sobm*sobt + cobm*cobt;                                           

      mat[0][0] = xx;
      mat[0][1] = yx;
      mat[0][2] = zx;
      mat[1][0] = xy;
      mat[1][1] = yy;
      mat[1][2] = zy;
      mat[2][0] = xz;
      mat[2][1] = yz;
      mat[2][2] = zz;

      return;

}




/* ----------------------------------------------------------------
 *      subroutine etilt (tjd,oblm,oblt,eqeq,dpsi,deps) 
 *                                                                               
 *    this subroutine computes quantities related to the orientation            
 *    of the earth's rotation axis at julian date tjd.                          
 *                                                                              
 * the USNO comments:
 *         tjd    = tdb julian date for orientation parameters (in)             
 *         oblm   = mean obliquity of the ecliptic in degrees at                
 *                  date tjd (out)                                              
 *         oblt   = true obliquity of the ecliptic in degrees at                
 *                  date tjd (out)                                              
 *         eqeq   = equation of the equinoxes in seconds of time at             
 *                  date tjd (out)                                              
 *         dpsi   = nutation in longitude in seconds of arc at                  
 *                  date tjd (out)                                              
 *         deps   = nutation in obliquity in seconds of arc at                  
 *                  date tjd (out)                                              
 * inputs
 * 	tjd, julian day of interest
 * outputs
 *	none
 * side effects
 * 	sets the five variables pointed to by the remaining calling parameters 
 * ----------------------------------------------------------------
 */                                                                               
                                                                               
void
etilt ( tjd, oblm, oblt, eqeq, dpsi, deps) 
     double tjd;	/* time, julian day, input */
     double *oblm;	/* mean obliquity */
     double *oblt;	/* true obliquity */
     double *eqeq;	/* equation of equinoxes in seconds of time */
     double *dpsi;	/* nutation in longitude */
     double *deps;	/* nutation in obliquity */
{

      double t0,t,t2,t3;
      double seccon,obm,obt,ee;
      double psi,eps;	/* nutations in longitude, obliquity */                                  
                                                                               
      t0 = 2451545.0;	/* t0 = tdb julian date of epoch j2000.0   */
      seccon = 206264.8062470964;                                          
                                                                               
      t = (tjd - t0) / 36525.0;                                                
      t2 = t * t;                                                                
      t3 = t2 * t;                                                               
                                                                               
      /* obtain nutation parameters in seconds of arc */ 
      /* this next, nod_, with an underscore, is in both nod.f and nod.c */ 
      nod_( &t, &psi, &eps);  
      /* greennod(), in greennu.c gives the same to less accuracy: */ 
      /* gnod( &t, &psi, &eps ); */

      /* compute mean obliquity of the ecliptic in seconds of arc */ 
      obm = 84381.4480 - 46.8150*t - 0.00059*t2 + 0.001813*t3;                                                       
                                                                               
      /* compute true obliquity of the ecliptic in seconds of arc */ 
      obt = obm + eps;
                                                                               
      /* compute equation of the equinoxes in seconds of time */ 
      ee = psi / 15.0 * cos (obt/seccon);                                     
                                                                               
      /* convert obliquity values to degrees */ 
      obm = obm / 3600.0;                                                      
      obt = obt / 3600.0;                                                      
                                                                               
      *oblm = obm;                                                                
      *oblt = obt;                                                                
      *eqeq = ee;                                                                 
      *dpsi = psi;                                                                
      *deps = eps;                                                                
                                                                               
      return;

}


/* ----------------------------------------------------------------
 * files formerly in greennu.c:
 * ----------------------------------------------------------------

/* ----------------------------------------------------------------
 * GreenNutations calculates nutation parameters from RM Green's formulas 
 * inputs
 *	unix time
 *	ptr to de,	nutation in obliquity
 *	ptr to dpsi, 	nutation in longitude
 *	ptr to true obliquity
 * outputs
 *	none
 * side effects
 *	sets the above pointed-to floating quantities
 * source
 *	RM Green, pp. 229.
 * 	Montenbruck, pp. 21-22 gives a slightly diff. version 
 * ----------------------------------------------------------------
 */

void
GreenNutations( utime, delta_psi, delta_e, true_epsilon )
    double utime;		/* unix time */
    double  *delta_psi;		/* the nutational variation of the pole orthogonal to de */ 
    double  *delta_e;		/* the nutational variation in obliquity of equator to orbital plane */  
    double  *true_epsilon;	/* obliquity of equator to eccliptic (orbital plane) */

{
    double dp;
    double de;
    double true_e;

    greennu( utime, &dp, &de, &true_e); 
    *delta_psi = dp;
    *delta_e   = de; 
    *true_epsilon = true_e; 
}


void
greennod( utime, dp, de )
     double utime; 
     double dp;
     double de;
{	
    double true_e;
    printf(" you called greennod; \n");
    printf(" you should call gnod() instead [with a different calling sequence] \n" ); 
    greennu( utime, &dp, &de, &true_e ); 
}


/* gnod() is to be a substitute for USNO NOVAS fortran routine nod(); 
 * units are arc-seconds 
 */ 

void 
gnod( T, dp, de )
     double *T; 	/* time in Julian centuries from 2000.0 (see nod.f) */
     double *dp;	/* nutation in longitude; arc-seconds; cf nod.f */
     double *de;	/* nutation in obliquity; arc-seconds; cf nod.f */
{
    double true_e;
    double utime;

    utime = calendarJulianToUnix( 2451545.0 + (*T)*36525.0 ); 

    greennu( utime, dp, de, &true_e );

    *dp *= TODEGREES;
    *dp *= 3600.;	/* convert to arc-seconds */ 

    *de *= TODEGREES;
    *de *= 3600.;	/* convert to arc-seconds */ 


}


/* note output units are RADIANS RADIANS RADIANS RADIANS RADIANS RADIANS  */ 

void
greennu( utime, delta_psi, delta_e, true_epsilon )
    double    utime;		/* unix time */
    double   *delta_psi;	/* nutational variation of the pole orthogonal to de */ 
    double   *delta_e;		/* nutational variation 
				   in obliquity of equator to orbital plane */  
    double   *true_epsilon;	/* obliquity of equator to eccliptic (orbital plane) */
{
    /* nutational quantities: */ 
  double  de;			/* nutational variation 
				   in obliquity of equator to orbital plane */  
  double  dpsi;		/* the nutational variation of the pole orthogonal to de */ 
  double  epsilon;		/* obliquity of equator to eccliptic (orbital plane) */

    /* times: */ 
    double jd;			/* a julian date */
    double T;			/* julian centuries from J 2000.0 */

    /* lunar and solar orbital quantities: */
    jd = calendarUnixToJulian( utime );
    T = (jd - 2451545.0) / 36525.; 

    fundamentalArguments( jd ); 	/* this is in sunmoon.c */ 
    
    /* RM Green, p. 229; units are arc-seconds: */
    dpsi = - 17.1996 * sin(meanMoonNode) 
      - 1.3187 * sin(2.*(meanMoonArgument - meanElongation + meanMoonNode))
      - 0.2274 * sin(2.*(meanMoonArgument - meanMoonNode));

    de   = 9.2025 * cos(meanMoonNode)
      + 0.5736 * cos(2.*(meanMoonArgument - meanElongation + meanMoonNode))
      + 0.0927 * cos(2.*(meanMoonArgument - meanMoonNode));

    dpsi /= 3600.;	/* convert to degrees */
    dpsi *= TORADIANS;	/* then to radians */

    de   /= 3600.;	/* convert to degrees */
    de   *= TORADIANS;	/* then to radians */

    /* I take the obliquity from p. 220, eqn. (9.25): */
    epsilon = 23.43929111;	/* degrees */
    /* these terms are in arc-seconds: */
    epsilon += ( - 46.815*T - 0.001*T*T + 0.002*T*T*T)/3600.;	
    epsilon *= TORADIANS;
    epsilon += de;		/* I am GUESSING that de should be added to epsilon */

    *delta_e		= de;		
    *delta_psi		= dpsi;		
    *true_epsilon	= epsilon;	

}


/* ----------------------------------------------------------------
 * files from equinox.c, mosty for precession 
 * ----------------------------------------------------------------
 */ 

/* 
 * equinox.c contains routines for changes in coordinate system from J2000.0 to date
 * these routines are called for test, print, 
 *	and demonstration purposes from almanacTest.c
 * 88/04/07
 * 88/08/18
 * 89/05/18
 * 89/06/04 ff.
 */


/* ----------------------------------------------------------------
 * almanacPrecession gives precession matrix as function of unix-time
 * ----------------------------------------------------------------
 */

void
almanacPrecession( utime, precession )
     double utime;			/* unix-time */ 
     matrix3d precession;		/* the answer, the precession matrix */ 
{
    double jdnow;
    double zeta, z, theta;

    jdnow = calendarUnixToJulian(utime);
 
#ifdef rmGreen
    rm_green_zeta( utime, &zeta, &z, &theta);
#endif rmGreen
    usnoPrecess( 2451545.0, jdnow, &zeta, &z, &theta );

    precessionMatrix( zeta, z, theta, precession );

}






/* ----------------------------------------------------------------
 * precessionMatrix computes the precession matrix, given zeta, z, theta
 * inputs
 *	zeta
 *	z
 *	theta
 * 	place to put matrix
 * outputs
 * 	none
 * side effects
 * 	puts matrix where you tell it to
 * ----------------------------------------------------------------
 */

void
precessionMatrix( zeta, z, theta, precession )
     double zeta;		/* precessional quantities */
     double z;
     double theta;
     matrix3d precession;	/* the rotation matrix for precession to be found */
{
    double szeta, czeta; /* trig functions of zeta */
    double sz, cz;	/* trig functions of z */
    double stheta, ctheta; /* trig functions of theta */ 

    szeta  = sin(zeta);
    czeta  = cos(zeta);
    sz     = sin(z);
    cz     = cos(z);
    stheta = sin(theta);
    ctheta = cos(theta);

    /* RM Green, p. 221-222: */
    precession[0][0] = - szeta*sz + czeta*cz*ctheta;
    precession[0][1] = - czeta*sz - szeta*cz*ctheta;
    precession[0][2] = - cz*stheta;

    precession[1][0] =   szeta*cz + czeta*sz*ctheta;
    precession[1][1] =   czeta*cz - szeta*sz*ctheta;
    precession[1][2] = - sz*stheta;

    precession[2][0] =   czeta*stheta;
    precession[2][1] = - szeta*stheta;
    precession[2][2] =   ctheta;

}



/* ----------------------------------------------------------------
 * almanacGreenNutation gets nutation numbers per RM Green's formulas
 *	and creates a nutation matrix
 * inputs
 *	unix time
 *	matrix 
 * outptus
 *	none
 * side effects
 *	sets the matrix you gave it 
 *----------------------------------------------------------------
 */

void
almanacGreenNutation( utime, nutation )
     double utime;		/* unix time */
     matrix3d nutation;		/* nutation matrix to be created */
{
    /* nutational quantities: */ 
    double  de;			/* the nutational variation 
				   in obliquity of equator to orbital plane */  
    double  dpsi;		/* the nutational variation 
				   of the pole orthogonal to de */ 
    double  epsilon;		/* obliquity of equator to eccliptic (orbital plane) */

    GreenNutations( utime, &dpsi, &de, &epsilon);	/* in greennu.c */
    nutationMatrix( dpsi, de, epsilon, nutation);

}



/* ----------------------------------------------------------------
 * nutationMatrix makes a transform matrix out of nutation parameters 
 * inputs
 *	dpsi, nutation in longitude
 *	de,   nutation in obliquity
 *	epsilon, total obliquity
 *	nutation, a matrix to put the answers in 
 * outputs
 *	none
 * side effects
 *	sets the nutation matrix
 * sources
 *	RM Green, p. 232;
 *	Lunar and planetary Coordinates, p. 320.
 * ----------------------------------------------------------------
 */

void
nutationMatrix( dpsi, de, epsilon, nutation)
     double dpsi;	/* nutation in longitude */
     double de;		/* nutation in obliquity */
     double epsilon;	/* true obliquity */
     matrix3d nutation;	/* the nutation matrix to be created */
{

    nutation[0][0] = 1.;	
    nutation[0][1] = - dpsi * cos(epsilon);
    nutation[0][2] = - dpsi * sin(epsilon); 

    nutation[1][0] = - nutation[0][1];
    nutation[1][1] = 1.;
    nutation[1][2] = - de;

    nutation[2][0] = - nutation[0][2];
    nutation[2][1] = - nutation[1][2];
    nutation[2][2] = 1.;

}




/* ----------------------------------------------------------------
 * RM Green's formulas for precessional quantities 
 * inputs
 *	time
 *	ptrs to zeta, z, theta
 * outputs
 *	none
 * side effects
 *	sets zeta, z, theta
 * method
 *	see RM Green, p. 219, eqn. (9.23)
 * ----------------------------------------------------------------
 */

void
rm_green_zeta( utime, zeta, z, theta )
     double utime;	/* unix time, seconds since 1970.0 */
     double *zeta;
     double *z;
     double *theta;
{
    double day;
    double t;

    day = calendarUnixToJulian( utime) ;	/* Julian day corresp. to utime */
    t = (day - 2451545.0 ) / 36525.;		/* time in centuries from J 2000.0 */

    /* RM Green, p. 219; units are degrees: */
    *zeta  = 0.6406161*t + 0.0000839*t*t + 5.0e-6*t*t*t;
    *z     = 0.6406161*t + 3.041e-4*t*t  + 5.1e-6*t*t*t;
    *theta = 0.5567530*t - 1.185e-4*t*t  - 1.1e-6*t*t*t;

    *zeta  *= TORADIANS;
    *z     *= TORADIANS;
    *theta *= TORADIANS;

}

