/* 
 * printers.c, routines to print various things 
 * 88/07/14 - 94/09/07 ff. 
 * Andrew P. Porter
 */

#include <stdio.h>
#include <math.h>

#include "../include/ms.h"
#include "../include/matrix.h"
#include "../include/hnav.h"



extern char *malloc();



/* ----------------------------------------------------------------
 * crash() is a surrogate for nrerror() 
 * ----------------------------------------------------------------
 */ 

void 
crash( complaint )
char *complaint;
{
	printf( "%s", complaint );
	exit(1);
}





/* ----------------------------------------------------------------
 * dumpOrbit prints canonical orbital elements, ref. mean anomaly, & ref. time
 * ----------------------------------------------------------------
 */

void 
dumpOrbit(orb)
     Orbit *orb;
{

    printf(" %5.4e ",orb->o_eccen);
    printf(" % 5.4e ",orb->o_periTime);
    printf(" %7.3f km ",orb->o_axis/1000.);
    printf(" % 6.5f ",orb->o_er.er_1);
    printf(" % 6.5f ",orb->o_er.er_2);
    printf(" % 6.5f, ",orb->o_er.er_3);
    printf(" % 5.4e % 5.4e ",orb->o_refAnomaly, orb->o_refTime);
    printf(" \n");

}


/* ----------------------------------------------------------------
 * dumpOrbit prints canonical orbital elements, ref. mean anomaly, & ref. time
 * ----------------------------------------------------------------
 */

void 
fdumpOrbit( fo, orb )
     FILE *fo; 
     Orbit *orb;
{

    fprintf( fo, " %5.4e ",orb->o_eccen);
    fprintf( fo, " % 5.4e ",orb->o_periTime);
    fprintf( fo, " %7.3f km ",orb->o_axis/1000.);
    fprintf( fo, " % 6.5f ",orb->o_er.er_1);
    fprintf( fo, " % 6.5f ",orb->o_er.er_2);
    fprintf( fo, " % 6.5f, ",orb->o_er.er_3);
    fprintf( fo, " % 5.4e % 5.4e ",orb->o_refAnomaly, orb->o_refTime);
    fprintf( fo, " \n");

}

/* ----------------------------------------------------------------
 * printOrbit does the same, in different units
 * 	angles in degrees; axis in kM; time is JD, not unix-time   
 * ----------------------------------------------------------------
 */ 

void 
printOrbit(orb)
     Orbit *orb;
{
    /* printf(" eTa,Oio;MT' " ); */ 

    printf(" %5.4e ",orb->o_eccen);
    printf(" % 12.2f ", 
	   calendarUnixToJulian(orb->o_periTime) );
    printf(" %7.3f km ",orb->o_axis/1000.);
    printf(" % 9.5f ",orb->o_er.er_1*TODEGREES);
    printf(" % 9.5f ",orb->o_er.er_2*TODEGREES);
    printf(" % 9.5f; ",orb->o_er.er_3*TODEGREES);
    printf(" %5.4e %12.2f ", 
	   orb->o_refAnomaly*TODEGREES, 
	   calendarUnixToJulian(orb->o_refTime) );
    printf(" \n");

}




/* ----------------------------------------------------------------
 * plotOrbit writes a file suitable for xgraph, plotting an orbit in x-y plane 
 * ----------------------------------------------------------------
 */

int
plotOrbit( orb, starttime, stoptime, name )
     Orbit *orb;
     double starttime;
     double stoptime;
     char *name; 	/* filename to write to for xgraph */ 
{
    Coord3d position; 
    double dt; 
    double time; 
    FILE *fo;

    fo = fopen( name, "w" ); 
    if (fo == NULL) 
    {
	printf("could not open file %s, in plotOrbit() \n", name ); 
	return 0; 
    }

    fprintf( fo, "\n%e %e\n\n", 0., 0. ); 	/* mark the focus */ 
    fprintf( fo, "\n%e %e\n\n", 0., 0. ); 	/* mark the focus */ 


    dt = (stoptime - starttime) / 120.; 
    for( time=starttime; time<=stoptime; time += dt )
    {
	navKepler( orb, time, &position  ); 
	fprintf( fo, "%e %e\n", position.x/AU, position.y/AU ); 
    }

    return 1; 

}

int
plotVeloc( orb, starttime, stoptime, name )
     Orbit *orb;
     double starttime;
     double stoptime;
     char *name; 	/* filename to write to for xgraph */ 
{
    Coord3d velocity; 
    double trueAnomaly; 
    double dt; 
    double time; 
    FILE *fo;

    fo = fopen( name, "w" ); 
    if (fo == NULL) abort(); 

    fprintf( fo, "\n%e %e\n\n", 0., 0. ); 	/* mark the focus */ 
    fprintf( fo, "\n%e %e\n\n", 0., 0. ); 	/* mark the focus */ 

    dt = (stoptime - starttime) / 20.; 
    for( time=starttime; time<=stoptime; time += dt )
    {
	trueAnomaly = navTrueAnomaly( orb, time );
	navVelocity( orb, trueAnomaly, &velocity ); 
	fprintf( fo, "%e %e\n", velocity.x, velocity.y ); 
    }

}


int
plotAnomalies( orb, starttime, stoptime, name )
     Orbit *orb;
     double starttime;
     double stoptime;
     char *name; 	/* filename to write to for xgraph */ 
{
    double trueAnomaly, eccAnomaly, meanAnomaly; 
    double dt; 
    double time; 
    FILE *fo;

    if ( parabola( orb )  ) return; 

    fo = fopen( name, "w" ); 
    if (fo == NULL) abort(); 

    fprintf( fo, "\n"); 
    dt = (stoptime - starttime) / 20.; 
    for( time=starttime; time<=stoptime; time += dt )
    {
	meanAnomaly = navMeanAnomaly( orb, time ); 
	eccAnomaly = navMeanToEccentricAnomaly( orb->o_eccen, meanAnomaly ); 
	trueAnomaly = navEccentricToTrueAnomaly( orb->o_eccen, eccAnomaly ); 

	fprintf( fo, "%e %e\n", time, trueAnomaly ); 
    }

    fprintf( fo, "\n"); 
    for( time=starttime; time<=stoptime; time += dt )
    {
	meanAnomaly = navMeanAnomaly( orb, time ); 
	eccAnomaly = navMeanToEccentricAnomaly( orb->o_eccen, meanAnomaly ); 
	trueAnomaly = navEccentricToTrueAnomaly( orb->o_eccen, eccAnomaly ); 

	fprintf( fo, "%e %e\n", time, eccAnomaly ); 
    }

    fprintf( fo, "\n"); 
    for( time=starttime; time<=stoptime; time += dt )
    {
	meanAnomaly = navMeanAnomaly( orb, time ); 
	eccAnomaly = navMeanToEccentricAnomaly( orb->o_eccen, meanAnomaly ); 
	trueAnomaly = navEccentricToTrueAnomaly( orb->o_eccen, eccAnomaly ); 

	fprintf( fo, "%e %e\n", time, meanAnomaly ); 
    }

}


/* ----------------------------------------------------------------
 * testRVOrbit tests conversions between r,v and orbit elements 
 * inputs
 *	none 
 * outputs
 *	number of errors / unacceptable results 
 * side effects
 *	only prints
 * results
 *	when last run, this produced max discrepancies of 44 meters
 *	between chosen orbits and orbits recovered from pos,vel 
 *	but for a polar orbit, with pos,vel at the pole, error was 1.e-18 meters . . .
 * ----------------------------------------------------------------
 */

int 
testRVOrbit()
{
    Orbit orb;		/* orbit as chosen */
    double time;	/* an arbitrary time, "now" */
    int errorcount;	/* number of unacceptable results */ 
    int  error; 	/* max deviation of calc orbit from orig orbit */ 

    FILE *fo;
    
    fo = fopen( "testRVOrbit.results", "w" ); 
    if (fo == NULL)
    {
	printf(" could not open file in testRVOrbit \n");
	return 1; 
    }

    fprintf( fo, 
      " testRVOrbit tests navPosVelToOrbit() and navKepler() and navVelocity() \n"); 
    error = 0; 
    time = 10000.;

    fprintf( fo, "\n ------------------------------------------ parabolas \n"); 
    navOrbitSet( &orb, E_MASS, 1.0, 0., 4.*6681000., 0., 0., 0., 0., 0. );
    error += RVOrbitTester( fo,  &orb, time);

    plotOrbit( &orb, -10000., 10000., "xparpos" ); 
    plotVeloc( &orb, -10000., 10000., "xparvel" ); 

    navOrbitSet( &orb, E_MASS, 1.0, 0., 4.*6681000., 0., 1., 0., 0., 0. );
    error += RVOrbitTester( fo,  &orb, time);

    navOrbitSet( &orb, E_MASS, 1.0, 0., 4.*6681000., 1., 1., 1., 0., 0. );
    error += RVOrbitTester( fo,  &orb, time);

    navOrbitSet( &orb, E_MASS, 1.0, 0., 4.*6681000., 0., PIhalf, 0., PIhalf, 0. );
    error += RVOrbitTester( fo,  &orb, time);


    fprintf( fo, "\n ------------------------------------------ ellipses \n"); 
    navOrbitSet( &orb, E_MASS, 0.0, 0., 4.*6681000., 0., 0., 0., 0., 0. );
    error += RVOrbitTester( fo,  &orb, time);

    plotOrbit( &orb, -10000., 10000., "xellpos" ); 
    plotVeloc( &orb, -10000., 10000., "xellvel" ); 

    navOrbitSet( &orb, E_MASS, 0.1, 0., 4.*6681000., 0., 1., 0., 0., 0. );
    error += RVOrbitTester( fo,  &orb, time);

    navOrbitSet( &orb, E_MASS, 0.5, 0., 4.*6681000., 1., 1., 1., 0., 0. );
    error += RVOrbitTester( fo,  &orb, time);

    navOrbitSet( &orb, E_MASS, 0.7, 0., 4.*6681000., 0., PIhalf, 0., PIhalf, 0. );
    error += RVOrbitTester( fo,  &orb, time);

    navOrbitSet( &orb, E_MASS, 0.7, 0., 4.*6681000., PIhalf, PI, PIhalf, PIhalf, 0. );
    error += RVOrbitTester( fo,  &orb, time);

    navOrbitSet( &orb, E_MASS, 0.999, 0., 4.*6681000., 1., 1., 1., PIhalf, 0. );
    error += RVOrbitTester( fo,  &orb, time);



    fprintf( fo, "\n ------------------------------------------ hyperbolas \n"); 
    
    navOrbitSet( &orb, E_MASS, 1.001, 0., -4.*6681000., 1., 1., 1., 0., 0. );
    error += RVOrbitTester( fo,  &orb, time); 

    navOrbitSet( &orb, E_MASS, 2.0, 0., -4.*6681000., 0., 0., 0., 0., 0. );
    plotOrbit( &orb, -10000., 10000., "xhyppos" ); 
    plotVeloc( &orb, -10000., 10000., "xhypvel" ); 

    time = 0.;
    error += RVOrbitTester( fo,  &orb, time);
    time = 10.;
    error += RVOrbitTester( fo,  &orb, time);
    time = -10.;
    error += RVOrbitTester( fo,  &orb, time);
    time = 10000.;
    error += RVOrbitTester( fo,  &orb, time);
    time = -10000.;
    error += RVOrbitTester( fo,  &orb, time);

    navOrbitSet( &orb, E_MASS, 3.0, 0., -4.*6681000., 0., 1., 0., 0., 0. );
    time = -10000.;
    error += RVOrbitTester( fo,  &orb, time);
    time = 0.;
    error += RVOrbitTester( fo,  &orb, time);

    navOrbitSet( &orb, E_MASS, 4.0, 0., -4.*6681000., 1., 1., 1., 0., 0. );
    time = 10000.;
    error += RVOrbitTester( fo,  &orb, time);
    time = 0.;
    error += RVOrbitTester( fo,  &orb, time);

    navOrbitSet( &orb, E_MASS, 5.0, 0., -4.*6681000., 0., PIhalf, 0., PIhalf, 0. );
    time = -10000.;
    error += RVOrbitTester( fo,  &orb, time);
    time = 0.;
    error += RVOrbitTester( fo,  &orb, time);

    return error; 
}


/* ----------------------------------------------------------------
 * RVOrbitTester returns max deviation of one orbit from another, 
 * within a limited range of the two orbits 
 * ----------------------------------------------------------------
 */ 


int    
RVOrbitTester( fo, orb, time)
     FILE *fo; 		/* where to write results/output to */ 
     Orbit *orb;	/* orbit as chosen */
     double time;	/* an arbitrary time, "now" */
{
    Orbit orbx;		/* orbit as determined from pos, vel */
    Coord3d pos;	/* position */ 
    Coord3d vel;	/* velocity */
    double trueAnomaly;	/* true anomaly */
    double error; 
    int ierror; 	/* whether the result is an error or not */ 

    fprintf( fo, "\n"); 

    trueAnomaly = navTrueAnomaly( orb, time);
    navAnomalyToPosition( orb, trueAnomaly, &pos);
    navVelocity( 	  orb, trueAnomaly, &vel);

    fprintf( fo, " orbit as chosen:    ");
    fdumpOrbit( fo, orb );

    navPosVelToOrbit( E_MASS, time, &pos, &vel, &orbx);
    fprintf( fo, " orbit as recovered: ");
    fdumpOrbit( fo, &orbx );
    error = howBad( orb, &orbx);
    fprintf( fo, " max deviation of determined orbit from chosen orbit = %e \n", error );

    ierror = 0;
    if (error  >  ABS(orb->o_axis) * 1.e-8 )
      ierror = 1; 

    return ierror; 

}


/* ---------------------------------------------------------------- */ 
/* these are associated with routines in matrix.c: */ 
/* ----------------------------------------------------------------
 * dumpVec prints the elements of a vector, and its LENGTH 
 * inputs
 *	ptr to Coord3d 
 *	comment string
 * outputs
 *	none
 * side effects
 *	only prints
 * ----------------------------------------------------------------
 */ 

void
dumpVec( v, s )
     Coord3d *v;
     char *s;
{
    printf(" %20s % 16e % 16e % 16e len: % 16e \n", s, v->x, v->y, v->z, GeoVectorLength(v));
}


/* ----------------------------------------------------------------
 * dumpNearUnitMatrix dumps a near-unit matrix
 * 	well, not quite: it prints the DIFFERENCE between it and a unit matrix
 * inputs
 *	label string
 *	the matrix to dump
 * outputs
 *	none
 * side effects
 *	only printf's
 * this routine seems to have originated in almanac/celcoord.c 
 *	moved to here 92/09/07
 * ----------------------------------------------------------------
 */

void
dumpNearUnitMatrix( s, m )
     char *s;		/* label string */
     matrix3d m;	/* matrix to be dumped */
{
    static matrix3d unit = {{1., 0., 0.}, {0., 1., 0.}, {0., 0., 1.}};
    int i,j;
    matrix3d d;

    matDiff( m, unit, d );

    printf(" %14s ", s );
    for ( i=0; i<3; ++i) 
      for ( j=0; j<3; ++j)
	printf(" % 11.8f", d[i][j] );
    printf("\n");

}
    

/* ----------------------------------------------------------------
 * dumpQuat prints the elements of a quaternion 
 * inputs
 *	ptr to quaternion
 *	comment string
 * outputs
 *	none
 * side effects
 *	only prints
 * ----------------------------------------------------------------
 */ 

void
dumpQuat(q,s)
     Quaternion *q;
     char *s;
{
    double sinx;			/* sin( theta/2), theta is the rotation angle of the quaternion */
    double x, y, z, w;		/* components of the quaternion */
    double theta;		/* the rotation angle of the quaternion */ 

    x = q->q1;
    y = q->q2;
    z = q->q3;
    w = q->q4;

    /* print the quat as it is, <vector*sin(theta/2), cos(theta/2)> */
    printf(" %20s   % 14e % 14e % 14e % 14e \n", s, x, y, z, w);

    /* print the quat as <vector, theta> */
    sinx = x*x + y*y + z*z;
    theta = 2. * atan2( sinx, w*w ); 
    if (sinx != 0.)
    {
	sinx = sqrt(sinx);
	x /= sinx;
	y /= sinx;
	z /= sinx;
    }
    w = 2. * acos( w);

    printf("   un-normalized:       % 14e % 14e % 14e % 14e \n", x, y, z, w);

}

/* ----------------------------------------------------------------
 * matDump prints matrix elements
 * inputs
 *	3x3 array
 *	comment or label string
 * outputs
 *	none
 * side effects
 *	only prints
 * ----------------------------------------------------------------
 */
 
void
matDump(p,s)
     double p[3][3];
     char *s;
{
  int i; 

  printf("%s\n",s);
  for (i=0; i<3; ++i) 
    printf(" %e %e %e\n",p[i][0], p[i][1], p[i][2]);

}


/* ----------------------------------------------------------------
 * matFDump prints matrix elements to a disk file
 * inputs
 *	file descriptor
 *	3x3 array
 *	comment/label string
 * outputs
 *	none
 * side effects
 *	only prints to disk file 
 * ----------------------------------------------------------------
 */
 
void
matFDump(fo,p,s)
     FILE *fo;
     double p[3][3];
     char *s;
{
  int i; 

  fprintf( fo, "%s\n", s );
  for (i=0; i<3; ++i) 
    fprintf( fo, " %e %e %e\n", p[i][0], p[i][1], p[i][2] );

}
