
/* ----------------------------------------------------------------
 * rwod.c: short for <read write o-div >
 *	O Division being where this code was developed 
 *		at the Lawrence Livermore National Laboratory 
 *	routines to read and write jpl-ephmeris in odiv format 
 * 		by a. porter, 91/12/01  ff.
 * 	to read an ascii image of a JPL ephemeris tape 
 *	in O-Division proprietary (eye-friendly) format 
 * ----------------------------------------------------------------
 */

#include <stdio.h>
#include <stdlib.h>
#include <ctype.h>
#include <math.h>
#include <string.h>

#include "../include/nrutil.h"

#include "../include/ms.h" 
#include "../include/matrix.h"
#include "../include/hnav.h"   




/* ----------------------------------------------------------------
 * getTellInfo reads an ascii file of ftell results on a binary ephemeris file
 *	for use in calls to fseek in later reads of the same file 
 *	(look at a .tel file to see what this routine reads;
 *	a .tel file is an index to a binary data file, and this routine
 *	retrieves that index-information)
 * inputs 
 *	tell-file name 
 * 	three arrays in which to put data: 
 * side effects
 *	fills starts[], ends[], tells[]
 * outputs 
 *	record count for success, 0 for failure 
 * ----------------------------------------------------------------
 */


int 
getTellInfo( tell_file, starts, ends, tells )
     char *tell_file;	/* the file of ftell index info to be read */ 
    double starts[];	/* array of start dates */ 
    double ends[];	/* array of end dates */ 
    long int tells[]; 	/* array of ftell results, for fseek */ 
{
    int i;	/* loop control */ 
    int j; 
    char buf[128]; 
    long int where; 
    double jdStart;
    double jdEnd;
    static FILE *ft; 
    int success; 
    int ierror;		/* error count */ 
    int count;		/* count of records in the .tel file */ 



    if (ft != NULL) fclose( ft ); 
    ft = getfile( tell_file, "r" );
    if (ft == NULL)
    {
	printf(" could not open tell file %s \n", tell_file );
	return 0; 
    }

    for (i=0; i<TellTableSize; ++i)
    {
	starts[i] = ends[i] = 0.; 
	tells[i] = 0;
    }

    for( i=0; i<TellTableSize; ++i)
    {
	if ( fgets( buf, 80, ft ) == NULL)
	{
	    break; 
	}
	count = i; 
	success = sscanf( buf, "%d jdStart %lf jdEnd %lf %lx", 
			 &j, &jdStart, &jdEnd, &where ); 
	if (success != 4)
	  printf(" line :%s:, %d, %12.2f, %12.2f, %lx \n", 
		 buf, j, jdStart, jdEnd, where );
	starts[i] = jdStart;
	ends[i] = jdEnd;
	tells[i] = where; 
    }
    ++count; 	

    ierror = 0; 
    for (i=0; i<count; ++i)
    {
	if ( starts[i] >= ends[i] )
	{
	    printf(" error in getTellInfo : %d %12.2f, %12.2f, %x \n", 
		   i, starts[i], ends[i], tells[i] ); 
	    ++ierror;
	}
	    
	if ( i >= 1   &&    tells[i] <= tells[i-1] ) 
	{
	    printf(" error in getTellInfo: %d tells[%d] %x, tells[%d] %x \n", 
		   i, i, tells[i], i-1, tells[i-1]  ); 
	    printf("                dates: %d %12.2f, %12.2f, %x \n", 
		   i, starts[i], ends[i], tells[i] ); 
	    ++ierror;
	}
    }
    if (ierror != 0)
    {
	printf(" ierror in getTellInfo = %d \n", ierror );
	exit(1); 
    }

    if (count <= 0)
    {
	printf(" very bad count = %d \n", count );
	exit(1); 
    }

    return count; 

}


/* ----------------------------------------------------------------
 * GetHeapByFtell reads a binary ephemeris file 
 * 	and returns a filled heap of data 
 * inputs
 *	header file name 	tells where the header info is 
 * 	data file name 		tells where to find the fseek-ftell arguments 
 * 	file pointer to data file 
 * 	julian date for which heap is requested 
 * outputs 
 *	pointer to a filled heap, or NULL, on failure 
 * side effects 
 *	file activity, fills heap 
 * ----------------------------------------------------------------
 */ 


TapeHeap * 
GetHeapByFtell( headername, datafile, fi, jd )
     char *headername;		/* name of header file, if it is known already */ 
     char *datafile;		/* name of data file opened at fi */ 
     FILE *fi; 			/* file in which to look for ephemeris record */ 
     double jd; 		/* the date for which want ephemeris record */ 
{
    static TapeHeap 	*heap=NULL;	/* where to put the answers */  
    static TapeHeap	heapbuf;	
    int 		success=0;	/* success or failure */ 
    static double 	jdStart=0;	/* start julian date of a data record */
    static double 	jdEnd=0;	/* end date of same */ 
    char 		tell_file[128]; 
    static double	starts [TellTableSize];	/* array of start dates */ 
    static double	ends   [TellTableSize];	/* array of end dates */ 
    static long int	tells  [TellTableSize];	/* ftell results, for fseek */ 
    int j, i; 
    int recordSize; 
    static int saveSize; 
    char *s; 
    static int firsttime=1;
    static FILE *fl; 
    int recordCount; 	/* count of records in this data file */ 
    
    FILE *fo;


    if (firsttime)
    {
	fl = logfile();
	firsttime = 0;
    }


    /* get a heap, w/o its data yet   */
    heap = getHeaderHeap( headername, fi );
    if (heap == NULL)
    {
	printf(" could not get an empty heap in GetHeapByFtell \n");
	abort(); 
	return (TapeHeap *) NULL;
    }
    recordSize = getRecordLength( &heap->groupSeven ); 

    strcpy( tell_file, datafile ); 
    recordSize = getRecordLength( &heap->groupSeven ); 

    s = strstr( tell_file, ".bin" );
    if ( s == NULL ) 
    {
	printf(" in GetHeapByFtell \n" ); 
  	printf(" filename %s should have a .bin extension (it doesn't)! \n", 
	       tell_file ); 
	abort(); 
	return (TapeHeap *) NULL;
    }
    strcpy( s, ".tel" ); 


    recordCount = getTellInfo( tell_file, starts, ends, tells );
    if (recordCount <= 0)
    {
	printf(" could not get tell file in GetHeapByFtell \n");
	abort(); 
	return (TapeHeap *) NULL;
    }

    recordSize = getRecordLength( &heap->groupSeven ); 

    /* find the right data record */ 
    if (jd < starts[0] || jd > ends[recordCount-1] )
    {		/* jd requested is not in this data file */ 
	printf(" jd %12.2f is not in this data file( %12.2f to %12.2f ) \n",
	       jd, starts[0], ends[recordCount-1] ); 
	abort(); 
	return heap; 
    }

    for( j=0; j<recordCount; ++j)
    {
	if (jd >= starts[j] && jd <= ends[j] ) break;
    }

    /* yrrrch!  this (sometimes) gets around a hideous bug: */ 
    if ( strstr( datafile, ".bin") != NULL)
      fi = freopen( datafile, "rb", fi ); 
    else
      fi = freopen( datafile, "r", fi ); 

    recordSize = getRecordLength( &heap->groupSeven ); 

    success = fseek( fi, tells[j], 0 ); 
    
    if (success != 0) 	/* note: fseek returns 0 on success */ 
    {
	printf("\n\n could not position binary file in GetHeapByFtell \n");
	printf("  for jd %12.1f \n", jd ); 
	printf(" header name = %s, data file = %s, fi %x, jd %12.2f \n",
	       headername, datafile, fi, jd ); 
	heap = NULL; 

	return (TapeHeap *) NULL; 
    }

    /* now fill in the data record */ 
    success = fread( &(heap->data[0]), sizeof( double ), recordSize, fi ); 



    heap->recordCount = recordCount; 
    heap->jdStart = heap->data[0];
    heap->jdEnd   = heap->data[1]; 
    jdStart = heap->jdStart;
    jdEnd = heap->jdEnd; 
    if (success != recordSize ) 
    {
	printf("could not read exactly a full record in GetHeapByFtell \n" ); 
	printf(" for jd %12.1f \n", jd ); 
	printf("  count = %d, recordSize = %d \n", success, recordSize ); 
	printf(" j = %d, tell[j] = %x, data file %s, tell file %s \n", 
	       j, tells[j], datafile, tell_file ); 
	heap = NULL; 
	abort(); 
	return (TapeHeap *) NULL; 
    }
    

    return heap; 

}


/* ----------------------------------------------------------------
 * getHeaderHeap gets a data-record (a "heap") 
 *	with only the header information in it, 
 *	i. e., no chebyshev polynomial data coefficients 
 * inputs
 *	header-file name, if known; uses default ENV. variable otherwise
 *	ptr to data file
 * outputs
 *	ptr to a TapeHeap / data-record
 * side effects
 *	saves old TapeHeap, and returns ptr to it if still good
 *	fills in header info only in the data-record;
 * 	actual data is to be filled in elsewhere
 * 
 * used in GetHeapByFtell 
 * ----------------------------------------------------------------
 */


TapeHeap *
getHeaderHeap( headername, fi )
     char *headername; /* name of header file, if it is known already */ 
     FILE *fi;	       /* this identifies the data file (NOT the header file) */ 
{
    static TapeHeap *heap=NULL;	/* ptr to where to put the answers */  
    static TapeHeap heapbuf;	/* where to put the answers */  
    static FILE *old_fi=NULL;	/* from last call */
    char *filename; 		/* name of file with header-data */ 
    int success=1; 

    if ( heap != NULL && fi == old_fi ) 
      return heap; 	/* do nothing: the heap u still have is the one u want */ 

    if (heap != NULL) 
    {
	heap = (TapeHeap *) NULL;
    }
    if (headername != NULL   &&   strlen( headername ) > 0 ) 
      filename = headername; 
    else 
    {
	filename = getHeaderName() ; 
    }
    if (filename == NULL)
    {
	printf(" could not get header file name in GetHeapFromOdiv \n");
	return (TapeHeap *) NULL;
    }
    heap = &heapbuf;  
    success = fileToEmptyHeap( filename, heap );
    old_fi = fi; 		/* so we know we've seen this file before */ 
    
    if (success == 1) 
      return heap; 
    else 
      return (TapeHeap *) NULL;

}


/* ----------------------------------------------------------------
 * GetHeapFromOdiv gets the data groups from a tape in Odiv format, 
 *	and finds the data set containing the jd of interest 
 * inputs
 *	a FILE containing an ephemeris 
 *	a julian date of interest to you 
 * outputs
 *	ptr to a TapeHeap, the collection of answers 
 *	NULL, if failure 
 * side effects 
 * 	fills the TapeHeap that it creates 
 * NOTE NOTE NOTE NOTE 
 *	to get a heap from JPL-format, 
 *		you need to call GetHeapFromJPL() instead  
 *	(it is obsolete and no longer supported here; but it was in readjpl.c)
 * ----------------------------------------------------------------
 */ 


TapeHeap * 
GetHeapFromOdiv( catalog, headername, fi, jd )
     char *catalog;    /* the catalog file name through which we were called  */ 
     char *headername; /* name of header file, if it is known already */ 
     FILE *fi; 	       /* the file in which to look for an ephemeris record */ 
     double jd;        /* the date for which you want an ephemeris record */ 
{
    static TapeHeap heapbuf;    /* to hold the data record */ 
    static TapeHeap *heap=&heapbuf;	/* where to put the answers */  
    static char *filename;	/* name of file with header-data */ 

    int success=0;		/* success or failure */ 

    static double jdStart=0;	/* start julian date of a data record */
    static double jdEnd=0;	/* end date of same */ 

    if ( needHeap( catalog, heap, jd )  )  
    {
      if (headername != NULL   &&   strlen( headername ) > 0 ) 
	filename = headername; 
      else 
	{
	  filename = getHeaderName() ; 
	}
      success = fileToEmptyHeap( filename, heap );
      if (success != 1)
	{
	  printf(" in GetHeapFromOdiv, fileToEmptyHeap failed, filename = %s \n",
		 filename );
	  return (TapeHeap *) NULL;
	}
    }    /* we now have a heap, possibly left over from an earlier call  */


    /* find the right data record */ 
    success = positionOdivFile( fi, jd, &jdStart, &jdEnd );  
    if (success != 1) 
      {
	printf(" could not position Odiv file for jd %12.1f \n", jd ); 
	printf("  called for fi = %x, jd = %12.2f, ", fi, jd );
	printf(" returns with jdStart = %12.2f, jdEnd = %12.2f \n", 
	       jdStart, jdEnd ); 
	heap = NULL; 
	return (TapeHeap *) NULL; 
      }

    /* now fill in the data record */ 
    success = readRecord( fi, heap ); 
    if (success != 1) 
    {
	printf("could not read record in GetHeapFromOdiv for jd %12.1f \n", jd ); 
	heap = NULL; 
	return (TapeHeap *) NULL; 
    }
    
    return heap; 

}



/* ----------------------------------------------------------------
 * fileToEmptyHeap reads a header-file to an empty TapeHeap 
 *	this is for ODIV format !!!!
 * inputs
 *	filename
 *	a heap to load header data into 
 * outputs
 *	the created TapeHeap
 * side effects
 * 	opens and reads file named on input
 * 	fills the TapeHeap groups 1-7 from the file
 * 	only group 8, the heap->data[], is empty 
 *	closes input file given as its first argument  
 * ----------------------------------------------------------------
 */

int 
fileToEmptyHeap( filename, heap )
     char *filename;	/* the file to get the TapeHeap from */ 
     TapeHeap *heap;	/* where to put the answers */
{
    static FILE *fi;	 	/* for the file to read them from */ 
    char buf[120];	/* to read a line of input into */ 
    char *line; 
    int i, j, k;	/* loop control */ 
    int count;		/* tells count of args read */ 
    int groupNumber;	/* to be read from file */ 

    if (fi != NULL) fclose( fi ); 
    fi = fopen( filename, "r" );
    if (fi == NULL)
    {
	printf(" fileToEmptyHeap could not read file %s \n", filename );
	fclose( fi ); 
	return 0; 
    }

    /* in general, getEmptyHeap(fi) and its called routines 
       * will NOT work -- that is for jpl format !!!! 
       * we supply other code here 
       */

    /* get the "group 1" line, else fail */
    line = ephemline( fi ); 
    count = sscanf( line, "group %d", &groupNumber );
    if (count != 1 || groupNumber != 1)
    {
	printf(" fileToEmptyHeap could not find group-1 header line \n");
	fclose( fi ); 
	return 0;
    }
    for( k=0; k<OneSize; ++k)
      for(i=0; i<LONGlineLENGTH; ++i)
	heap->groupOne[k][i] = '\0';

    for( k=0; k<OneSize; ++k)
    {
	line = ephemline( fi );
	if (
	       line == NULL 
	    || strlen(line) == 0 
	    || strncmp( line, "group", 5) == 0 
	    ) 
	{
	    ungetEphemLine( line ); 
	    break; 
	}
	strncpy( heap->groupOne[k], line, 80 );
	heap->groupOne[k][80] = '\0';
	for(i=0; i<LONGlineLENGTH; ++i) 
	  if (heap->groupOne[k][i] == '\n') 
	    heap->groupOne[k][i] = '\0';
    }

    /* group 2 */ 
    line = ephemline( fi ); 
    count = sscanf( line, "group %d", &groupNumber );
    if (count != 1 || groupNumber != 2)
    {
	printf(" fileToEmptyHeap could not find group-2 header line \n");
	printf("line :%s:\n", line );
	fclose( fi ); 
	return 0; 
    }

    for( k=0; k<TwoSize; ++k)
      for(i=0; i<LONGlineLENGTH; ++i)
	heap->groupTwo[k][i] = '\0';

    for( k=0; k<TwoSize; ++k)
    {
	line = ephemline( fi );
	if (
	       line == NULL 
	    || strlen(line) == 0 
	    || strncmp( line, "group", 5) == 0 
	    ) 
	{
	    ungetEphemLine( line ); 
	    break; 
	}
	strncpy( heap->groupTwo[k], line, 80 );
	heap->groupTwo[k][80] = '\0';
	for(i=0; i<LONGlineLENGTH; ++i) 
	  if (heap->groupTwo[k][i] == '\n') 
	    heap->groupTwo[k][i] = '\0';
    }


    /* group 3 was not interesting and was left out */

    /* group 4 is different from jpl format */
    line = ephemline( fi ); 
    count = sscanf( line, "group %d", &groupNumber );
    if (count != 1 || groupNumber != 4)
    {
	printf(" fileToEmptyHeap could not find group-4 header line \n");
	fclose( fi ); 
	return 0; 
    }
    count = 0; 
    line = ephemline( fi ); 
    count += sscanf( line, " earliestDate %le", 
		    &(heap->groupFour.earliestDate) );
    line = ephemline( fi ); 
    count += sscanf( line, " latestDate %le", 
		    &(heap->groupFour.latestDate) );
    line = ephemline( fi ); 
    count += sscanf( line, " recordLength %le", 
		    &(heap->groupFour.recordLength) );
    line = ephemline( fi ); 
    count += sscanf( line, " recordInterval %le", 
		    &(heap->groupFour.recordInterval) );
    line = ephemline( fi ); 
    count += sscanf( line, " numberOfBodies %d", 
		    &(heap->groupFour.numberOfBodies) );
    line = ephemline( fi ); 
    count += sscanf( line, " versionNumber %le", 
		    &(heap->groupFour.versionNumber) );
    if ( count != 6 )
    {
	printf(" could not read group 4 data correctly \n" ); 
	fclose( fi ); 
	return 0; 
    }

    /* group 5 (and 6) */ 
    line = ephemline( fi ); 
    count = sscanf( line, "group %d", &groupNumber );
    if (count != 1 || groupNumber != 5)
    {
	printf(" fileToEmptyHeap could not find group-5 and 6 header line \n");
	fclose( fi ); 
	return 0; 
    }

    /* #### this next number, GROUP_SIX_SIZE, now 199, is hard-wired for de-200, 
     * and wrong for de-245;
     * header.245 has void entries to fill out 199 places,
     * but another ephemeris with _more_ than 199 will be up a creek .... 
     * and will I (or you, or someone else) notice that groups 5-6 
     *		have to be EXACTLY 199 entries ????
     * ####  */ 
    for( k=0; k<GROUP_SIX_SIZE; ++k)
    {
	line = ephemline( fi ); 
	sscanf( line, " %s %le \n", &heap->groupFive[k], &heap->groupSix[k] );
	count = sscanf( line, "group %d", &groupNumber );
	if ( count == 1 )
	{
	    break;	/* you've come to group 7; quit */ 
	}
	/* #### #### make sure this works with "Void" and not "void" ... !!! #### */ 
	if ( strstr( heap->groupFive[k], "Void" ) != NULL )
	  break; 
    }

    /* group 7 */ 
    rewind( fi ); 
    while( 1 )
    {
	line = ephemline( fi ); 
	if (line == NULL)
	{
	    printf(" fileToEmptyHeap could not find group-7 header line \n");
	    fclose( fi ); 
	    return 0; 
	}
	count = sscanf( line, "group %d", &groupNumber );
	if (count == 1 && groupNumber == 7)
	  break; 
    }

    /* we loop to numberOfBodies, not SEVENSIZE,
     * because the group-7 data may only go to the number of bodies ...
     * ... and we already know, from group-4 above,
     * what the numberOfBodies is.
     */ 
    for( k=0;  k < (heap->groupFour.numberOfBodies);   ++k )
    {
	line = ephemline( fi );
	count = sscanf( line, " groupSeven %d %d %d", 
	       &heap->groupSeven.first[k],
	       &heap->groupSeven.ncoeffs[k],
	       &heap->groupSeven.nintervals[k]
	       ); 
	if ( count != 3)
	{
	    printf(" bombed in group 7 in fileToEmptyHeap \n");
	    fclose( fi ); 
	    return 0; 
	}
    }

    /* zero the empty parts of this heap: */ 
    heap->jdStart = 0.;
    heap->jdEnd   = 0.;

    orderHeap( heap ); 

    fclose( fi ); 
    return 1; 
}



/* ----------------------------------------------------------------
 * dumpRecord writes an ephemeris record to a file,
 *	in our format, not jpl's 
 * inputs
 * 	file to write to 
 *	heap to write from 
 * outputs
 *	1 for success, 0 for failure 
 * side effects 
 *	writes to file 
 * ----------------------------------------------------------------
 */

int
dumpRecord( fo, heap )
     FILE	*fo;		/* file to write to */
     TapeHeap	*heap;		/* contains the record to write to it */ 
{
    int j;			/* loop control */ 
    int body;
    int subinterval;
    int component; 		/* x,y,z;  or  dpsi,deps */
    int start;			/* starting index in the heap->data[] array */ 
    int ncomponents;		/* number of components */ 


    fprintf( fo, " \n"); 
    fprintf( fo, "record %11.2f to %11.2f \n", heap->jdStart, heap->jdEnd ); 
    fprintf( fo, "; %s to ", fvfJulianToDate( heap->jdStart ) );
    fprintf( fo, " %s \n",   fvfJulianToDate( heap->jdEnd )   );

    /* bodies */
    for ( body=0;  body < heap->groupFour.numberOfBodies;  ++body )
    {
	fprintf( fo, " \n"); 
	fprintf( fo, "; body %d \n", body ); 
	/* sub-intervals */
	for ( subinterval=0;  
	     subinterval < heap->groupSeven.nintervals[body];  
	     ++subinterval )
	{
	    fprintf( fo, " \n"); 
	    fprintf( fo, "; sub-int %d \n", subinterval ); 
	    /* components:  x,y,z or dpsi, deps */ 
	    if (body == 11) ncomponents = 2;
	    else	    ncomponents = 3;
	    for ( component=0;  component < ncomponents;  ++component )
	    {
		fprintf( fo, " \n"); 
		/* the chebyshev polynomial coefficients themselves, in order:  */
		start = heap->index [body] [subinterval] [component]; 
		for (j=0; j<heap->groupSeven.ncoeffs[body]; ++j )
		{
		    fprintf( fo, " %22.15e\n",  heap->data[ start + j ]    );
		}
	    }
	}
    }
    fprintf( fo, "end of record %12.2f to %12.2f; ", 
	    heap->jdStart, heap->jdEnd ); 
    fprintf( fo, " %s", fvfJulianToDate( heap->jdStart ) );
    fprintf( fo, " to %s \n", fvfJulianToDate( heap->jdEnd ) );
    fflush( fo ); 
    return 1; 

}



/* ----------------------------------------------------------------
 * readRecord . . . . an inverse of dumpRecord()
 * inputs
 *	file to read from 
 *		file is in proprietary format, i. e., written by dumpRecord()  
 *              i. e., this is an o-div _ascii_ format file 
 *		file must be positioned to start of a record 
 *	heap to put data in 
 * 		heap must be "in order": heap->index[][][] has been set 
 *		heap-> groups 1-7 are present and filled 
 * outputs
 * 	1 for success, 0 for failure 
 * side effects 
 * 	reads the next data-record and puts data in heap
 * 		heap->data
 *		heap->jdStart, heap->jdEnd
 * ----------------------------------------------------------------
 */ 


int
readRecord( fi, heap ) 
     FILE *fi; 		/* file to read from */
     TapeHeap *heap;	/* contains a tapeheap, possibly empty of data */ 
{
    char *line; 	/* for input */ 
    int k; 		/* loop control */ 
    double jdStart;
    double jdEnd; 
    int count=0; 
    int recordSize;	/* size of heap->data[] array */ 


    /* check for "record . . . " line, get jd's */
    for (k=0; k<2048; ++k)
    {
	line = ephemline( fi ); 
	if (line == NULL) 
	  break; 
	count = sscanf( line, "record %le to %le", &jdStart, &jdEnd );	
	if (count < 2) 	
	  continue; 	/* go on to beginning of a record */ 
	else 
	  break;	/* found it, so exit loop */ 
    }
    if (count < 2) 
    {
	/* skip the complaints: ???? */
	/*
	 * printf(" could not find start of a record \n"); 
	 * printf(" sscanf fails on line:%s:\n", line ); 
	 */
	return 0; 	/* so return failure  */ 
    }

    heap->jdStart = jdStart;
    heap->jdEnd   = jdEnd; 
    heap->data[0] = jdStart;
    heap->data[1] = jdEnd; 

    recordSize = getRecordLength( &(heap->groupSeven) ); 	
    /* we already have jdStart and jdEnd */ 

    for (k=2; k<recordSize; ++k)
    {
	line = ephemline( fi ); 
	count = sscanf( line, " %le", &heap->data[k] );
	if (count != 1) 
	{
	    printf(" something is wrong reading file in readRecord; k = %d \n", 
		   k ); 
	    printf("%s", line ); 
 	    abort(); 
	}
    }

    return 1; 

}


/* ----------------------------------------------------------------
 * positionOdivFile positions a file in odiv-format to desired record 
 * inputs
 * 	many
 * outputs
 *	1 for success, 0 for failure 
 * side effects
 * 	positions file
 *	sets *jdStart, *jdEnd 
 * where called
 *      only once, from GetHeapFromOdiv
 * ----------------------------------------------------------------
 */ 

int  
positionOdivFile( fi, jd, jdStart, jdEnd )
     FILE *fi;   /* ptr to ascii data file in which to find start of record */ 
     double jd;           /* date for which a record is sought */ 
     double *jdStart;     /* to be set to jdStart for the record to be found */
     double *jdEnd;       /* to be set to jdEnd for the record to be found */
{
    char buf[121];
    int count; 
    long offset; 
    char *fgetsReport; 
    static int firsttime=1;

    /* printf( "\n entering positionOdivFile %12.4f \n", jd );    */
    /* if the desired record is back of where we are, rewind */ 
    if ( jd < *jdStart) 
      {
	rewind( fi ); 
      }

    while (1)
    {
	/* skip to next data-record */ 
	offset = ftell( fi ); /* save <ftell> at beginning of next line */ 
	fgetsReport = fgets( buf, 120, fi );
	if ( fgetsReport == NULL)
	{
	    printf(" unexpected read-failure ffin positionOdivFile()," );
	    printf(" for jd %12.4f \n", jd ); 
	    return 0; 	/* we ran out of lines to read */ 
	}

	/* read its jdStart, jdEnd  */
	count = sscanf( buf, "record %le to %le", jdStart, jdEnd ); 
	if (count < 2) continue; 	/* not start of a data-record */ 

	if ( jd > *jdEnd) 
	  continue; 
	break; 
    }
    /* others (i. e., readRecord() )  need to read this line, too: */ 
    /* 3rd arg = 0: --> seek relative to beginning of file; 
     * see man pages for fseek 
     */ 
    fseek( fi, offset, 0 ); 

    return 1; 
}

/* ----------------------------------------------------------------
 * dumpEmptyHeap writes groups 1-7 of a tapeheap to an output file 
 *	in an ascii format 
 * inputs
 *	file
 *	TapeHeap
 * outputs
 *	1 for success, 0 for failure 
 * side effects 
 *	writes to file 
 * ----------------------------------------------------------------
 */

int 
dumpEmptyHeap( fo, heap )
     FILE *fo;		/* descriptor for filename */
     TapeHeap *heap;	/* the heap to write to it */ 
{
    int i,k;		/* loop control */ 
 
    /* write it all out */ 
    fprintf( fo, ";  dumpEmptyHeap output \n"); 
    
    /* group 1 */ 
    fprintf( fo, "group 1 \n"); 
    for(k=0; k<10; ++k)
    {
	if (heap->groupOne[k][0] == '\0') break;
	fprintf( fo, "%s\n", heap->groupOne[k] ); 
    }
    fflush( fo ); 

    /* group 2 */ 
    fprintf( fo, "group 2 \n"); 
    for(k=0; k<10; ++k)
    {
	if (heap->groupTwo[k][0] == '\0') break;
	fprintf( fo, "%s\n", heap->groupTwo[k] ); 
    }
    fflush( fo ); 

    /* group 4 */ 
    fprintf( fo, "group 4 \n"); 
    fprintf( fo, " earliestDate %20.15e \n", heap->groupFour.earliestDate );
    fprintf( fo, " latestDate   %20.15e \n", heap->groupFour.latestDate );
    fprintf( fo, " recordLength %20.15e \n", heap->groupFour.recordLength ); 
    fprintf( fo, " recordInterval %20.15e \n", heap->groupFour.recordInterval );
    fprintf( fo, " numberOfBodies %d \n", heap->groupFour.numberOfBodies ); 
    fprintf( fo, " versionNumber %20.15e \n", heap->groupFour.versionNumber ); 

    /* groups 5 and 6 */ 
    fprintf( fo, "group 5 \n"); 
    for( k=0; k<heap->size56; ++k)
    {
	fprintf( fo, " %s %20.15e \n", heap->groupFive[k], heap->groupSix[k] );
    }
    fflush( fo ); 

    /* group 7  etc. */
    fprintf( fo, "group 7 \n"); 
    for( k=0; k<SEVENSIZE; ++k ) 
    {
	fprintf( fo, " groupSeven %d %d %d \n", 
		heap->groupSeven.first[k], 
		heap->groupSeven.ncoeffs[k], 
		heap->groupSeven.nintervals[k]
		);
    }

    fflush( fo ); 

    /* go home */ 
    return 1;

}




