/* sunmoon.c contains routines for sun and moon position
 *	of lesser accuracy than using the JPL ephemerides
 *	most of this was written 1988-1990
 */

#include <stdio.h>
#include <math.h>
#include <time.h>

#include "../include/ms.h"
#include "../include/matrix.h"
#include "../include/hnav.h"


extern char *malloc();
extern double normalize();



 

/* ----------------------------------------------------------------
 * fundamentalArguments sets the fundamental arguments, which are in "common" storage
 * inputs
 *	time, julian day number, converted from Unix time, that is, Universal Time 
 * 		i. e., not terrestrial dynamical time or ephemeris time
 *	I believe these formulas are actually for UT, not TDT: cf. Almanac, p. D46
 * outputs
 *	none
 * side effects
 *	sets the fundamental arguments for the moon and sun 
 * sources
 * 	Oliver Montenbruck, _Practical Ephemeris Calculations_, for the Moon;
 *		he is dependent on <Improved Lunar Ephemeris>, 1954,
 *		a presentation of E. W. Brown's theory of the moon 
 *	the sun arguments probably come from the Almanac or RM Green
 * accuracy
 *	these are used in MontenbruckMoon(), to get 15 arc-min accuracy for the moon 
 *	they are good in the 20th century, probably well into the 21st
 * ----------------------------------------------------------------
 */

     double   meanMoonLongitude; /* L' in Green's notation (and the Almanac, p. D2) */
     double   meanMoonNode;	/* Omega, in AstAlmanac88 */
     double   meanMoonArgument;	/* F, = L' - Omega  (measured from asc.node) */
     double   moonMeanAnomaly;	/* M' 		    (measured from perigee)  */
     double   sunMeanAnomaly;	/* M  */
     double   meanElongation;	/* D  */
     double   moonPerigeeArgument; /* Gamma', in AstAlmanac88, p. D2 */ 
     double   meanSunLongitude;  /* L, acc. to p. D2 of the almanac */
     double   Obliquity;	/* obliquity of equator and eccliptic */

void 
fundamentalArguments( jd )
     double jd;		/* julian date, converted from UT */
{
    double T;		/* time in Julian centuries from 1900/1/0 noon 
			   (Montenbruck, p. 79) */
    T  = (jd - 2415020.0 ) / 36525.;	/* centuries from 1900.0 NOTE, NOT J2000.0 */

  /* these numbers are in storage "common" to all the almanac.c routines
  /* from Montenbruck, p. 79: */

  meanMoonLongitude   = (270.434164 + 481267.883142*T - 0.001133*T*T)*TORADIANS;
  meanMoonLongitude   = dnormalize( meanMoonLongitude, 0., PI2 );

  meanMoonNode        = (259.183275 -   1934.142008*T + 0.002078*T*T)*TORADIANS;
  meanMoonNode	      = dnormalize( meanMoonNode, 0., PI2 );

  meanMoonArgument    = meanMoonLongitude - meanMoonNode;
  meanMoonArgument    = dnormalize( meanMoonArgument, 0., PI2 );

  moonMeanAnomaly     = (296.104608 + 477198.849108*T + 0.009192*T*T)*TORADIANS;
  moonMeanAnomaly     = dnormalize( moonMeanAnomaly, 0., PI2 );

  sunMeanAnomaly      = (358.475833 +  35999.049750*T - 0.000150*T*T)*TORADIANS;
  sunMeanAnomaly      = dnormalize( sunMeanAnomaly, 0., PI2 );

  meanSunLongitude    = (279.696678 +  36000.768925*T + 0.000303*T*T)*TORADIANS,
  meanSunLongitude    = dnormalize( meanSunLongitude, 0., PI2 );

  meanElongation      = meanMoonLongitude - meanSunLongitude; 
  meanElongation      = dnormalize( meanElongation, 0., PI2 );
                         
  moonPerigeeArgument = meanMoonLongitude - moonMeanAnomaly;
  moonPerigeeArgument = dnormalize( moonPerigeeArgument, 0., PI2 );

    /* centuries from J2000.0 !!!! NOT same as above !!!! */
  T  = (jd - 2451545.0 ) / 36525.;
  /* this formula is from RM Green, p. 220: */
  Obliquity	      = (23. + 26./60. + 21.448/3600.)
			 - (46.815/3600.)*T
			 - (0.001/3600.)*T*T
			 + (0.002/3600.)*T*T*T;
  Obliquity	     *= TORADIANS;

}

 
/* ----------------------------------------------------------------
 * Position returns position of the sun . . . 
 * method:
 *	crude series of Astronomical Almanac 1988, p. C-24
 *	same as Astronomical Almanac 1989, p. C-24
 *	NOTE: these formulas are in degrees; we convert to radians 
 * inputs
 *	time	NOTE units are SECONDS from beginning of year 1970 (unix time)
 *	pointer to where to put cartesian answers
 * output
 *	none
 * side effects
 *	puts equatorial polar coords of sun in *pos
 * 	NOTE units are METERS, not km, not AU. you must convert !!!! 
 * these are geocentric equatorial coords
 * 	p. C-24 does not say what coord system it is referred to; 
 *	agreement is good for pp. C 4-18, probably results are 
 *	in coord system of date (i. e., a non-inertial coord system)  
 * we do not correct for abberation 
 *	because p. C 24 claims to do this already (cf. p. C2, mean longitude )
 * ----------------------------------------------------------------
 */

void 
sunPosition( time, spos )
     double time;		/* unix time */
     Coord3d *spos; 		/* where to put equatorial cartesian position */ 
{
    PCoord equ;			/* equatorial polar coords: ra-dec */
    double temp;
    double jd; 
    
    jd = calendarUnixToJulian( time ); 

    sunRadec( jd, &equ );

    /* matPolarToRect( &equ, spos ); */
    temp  = equ.pc_r * cos(equ.pc_lat)*cos(equ.pc_long);

    spos->y = equ.pc_r * cos(equ.pc_lat)*sin(equ.pc_long);
    spos->z = equ.pc_r * sin(equ.pc_lat);
    spos->x = equ.pc_r * cos(equ.pc_lat)*cos(equ.pc_long);

}

void
sunRadec( jd, pos)
     double jd;		
     PCoord *pos;		/* geocentric equatorial coords of the sun, */
                                /* NOTE: equator and equinox of J 2000.0 */
{
    double unix_time;		/* indep.; time scale == UT1, i. e., GMT; 
				   see day_number */ 
                                /* this is UNIX time, seconds from start of 1970 */
                                /* i. e., time in SI seconds since January 1 0h UT1 */
  double day_number;		/* Julian Day - 2451545.0; 
				   i. e., days after J 2000.0  */
  double year;			/* the year i. e., 1989 */
  double mean_longitude;		/* (degrees) */
  double mean_anomaly;		/* (degrees) */
  double ecclip_longitude;	/* (degrees) */
  double ecclip_latitude;	/* (degrees) */

  double right_ascension;	/* (sidereal hours) */
  double declination;		/* (degrees) */
  double radius;			/* (AU) */
  double ecl, cos_e;		/* temps */
  double time;			/* time in seconds from start of 1988 */
  double d1988;			/* unix time for start of 1988 */
  double temp;		

    unix_time = calendarJulianToUnix( jd ); 

  d1988 = calendarJulianToUnix( 2447161.5 );
  time = unix_time - d1988;
  day_number = time/86400. - 4384.5; 	/* day - J 2000.0; 	*/
  fundamentalArguments( jd );

  /* day numbers (p. B 2-3) appear to start with 1 (not 0) at 0h Jan 1 
   * and the formulas of p. C-24 use day numbers (this seems weird to me) 
   */
  day_number += 1; 

  mean_longitude 	= 280.460 + day_number*0.9856474;
  mean_longitude	= dnormalize(mean_longitude,0.,360.);
  mean_longitude       *= TORADIANS;

  mean_anomaly   	= 357.528 + day_number*0.9856003;
  mean_anomaly		= dnormalize(mean_anomaly,0.,360.);
  mean_anomaly	       *= TORADIANS;

  ecclip_longitude	= mean_longitude 
    + (  1.915 * sin (     mean_anomaly) 
       + 0.020 * sin ( 2 * mean_anomaly) ) * TORADIANS;
  ecclip_longitude	= dnormalize ( ecclip_longitude, 0., PI2 );

  ecclip_latitude	= 0.;

  cos_e			= cos(Obliquity);
  ecl			= ecclip_longitude;
  right_ascension	= atan2(cos_e*sin(ecl),cos(ecl));
  right_ascension       = dnormalize (right_ascension, 0., PI2); /* already radians */
  temp 			= sin(Obliquity ) * sin( ecclip_longitude );
  declination		= asin ( sin(Obliquity ) * sin( ecclip_longitude ) );	

  radius		= 1.00014 
    -  0.01671 * cos(      mean_anomaly ) 
    -  0.00014 * cos( 2. * mean_anomaly );

  /* radius was correct here, 90/09/29 */
  pos->pc_r		= radius*AU; 
  pos->pc_long		= right_ascension;
  pos->pc_lat		= declination; 

}




/* ----------------------------------------------------------------
 * routines derived from MontenbruckMoon()
 *	almanacMoonPosition()
 *	moonRadec()
 *	moonposition()
 *	ecclipticmoon()
 * inputs
 *	see argument lists 
 * outputs 
 *	none, for any
 * side effects
 * 	sets the position of the declared type where you tell them to
 * comments
 *	almanacMoonPosition is not recommended: note the zero of the time argument
 * NOTE:  the POLAR routines take time argument as JULIAN DATE,
 *	the CARTESIAN routines take time in SECONDS (with diff. zeros of time)
 *
 * ----------------------------------------------------------------
 */

void almanacMoonPosition( time, pos )
     double time;	/* time in SI seconds since 1990.0 */
     Coord3d *pos;
{
    double jd;
    double time1990;

    time1990 = calendarJulianToUnix(2447892.5);
    jd = calendarUnixToJulian( time + time1990 );
    moonPosition( jd, pos );
}

void 
moonPosition( time, pos )
     double time;	/* UNIX time, zero is 1970.0 */
     Coord3d *pos;
{
    double jd;		/* julian date */ 
    PCoord lambet;	/* eccliptic polar coords */
    Coord3d ecc;	/* eccliptic cartesian coords */
    Coord3d eccx;
    EulerRot eulers;	/* the euler angles for transf of coords */
    matrix3d mat;	/* the matrix for transf of coords */

    jd = calendarUnixToJulian( time );
    ecclipticMoon( jd, &lambet );

    matPolarToRect( &lambet, &ecc ); 

    eulers.er_1 = 0.;
    eulers.er_2 = Obliquity;
    eulers.er_3 = 0.; 

    matEulerToMat( &eulers, mat);
    matMatVV( mat, &ecc, pos );

}



void moonRadec( jd, radec )
     double jd;		/* julian date */
     PCoord *radec;	/* where to put equatorial position in polar coords */ 
     /* note angles are in radians */
{
    Coord3d equXyz;	/* position, equatorial cartesian coords */ 

    moonPosition( calendarJulianToUnix(jd), &equXyz );

    matRectToPolar( &equXyz, radec );
    /* printf(" jd %12.1f, moon ra = %8.4f \n", jd, (radec->pc_long)*TODEGREES/15. ); */ 
    radec->pc_long = dnormalize( radec->pc_long, 0., PI2 );

}


void 
ecclipticMoon( jd, ecclip )
     double jd;		/* julian date */
     PCoord *ecclip;	/* where to put equatorial position in polar coords */ 
     /* note angles are in radians */
{
    fundamentalArguments( jd );
    MontenbruckMoon( ecclip ); 
}


/* ----------------------------------------------------------------
 * MontenbruckMoon returns eccliptic moon coords from Montenbruck's formulas
 * inputs
 *	julian date, calc from UT 
 *	ptr to a PCoord, where to put the answer
 * outputs
 *	none 
 * side effects
 *	pc long == lambda, pc lat == beta, pc r == radius from pi
 * source is Oliver Montenbruck, <Practical Ephemeris Calculations>
 *	from EW Brown's theory of the moon, from <Improved Lunar Ephemeris, 1952-1959>
 *	Oliver Montenbruck has taken the leading terms only
 * accuracy
 *	good to 0.25 degrees -- this routine will hit the lunar disk
 *	should work any time in the 20th century, perhaps into the next ;
 *	Montenbruck expands fundamental arguments elements around 1900
 * coord system: ecliptic and equinox of date (Montenbruck, p. 79)
 * ----------------------------------------------------------------
 */     

void 
MontenbruckMoon( eccliptic )
     PCoord *eccliptic;		/* the eccliptic position, in polar coords */
{
  double lambda;			/* eccliptic longitude */
  double beta;			/* eccliptic latitude  */
  double parallax;		/* "pi", angle subtended by earth at moon */

  double temp;

  /* Montenbruck, p. 80 */
  lambda = 22640.*sin( moonMeanAnomaly )
    +  769.*sin( 2.*moonMeanAnomaly )
    +   36.*sin( 3.*moonMeanAnomaly )
    -  125.*sin( meanElongation )
    + 2370.*sin( 2.*meanElongation )
    -  668.*sin( sunMeanAnomaly )
    -  412.*sin( 2.*meanMoonArgument )
    +  212.*sin( 2.*meanElongation - 2.*moonMeanAnomaly )
    + 4586.*sin( 2.*meanElongation - moonMeanAnomaly )
    +  192.*sin( 2.*meanElongation + moonMeanAnomaly )
    +  165.*sin( 2.*meanElongation - sunMeanAnomaly )
    +  206.*sin( 2.*meanElongation - sunMeanAnomaly - moonMeanAnomaly )
    -  110.*sin( sunMeanAnomaly + moonMeanAnomaly )
    +  148.*sin( moonMeanAnomaly - sunMeanAnomaly );
    
  lambda /= 3600.;
  lambda *= TORADIANS;
  lambda += meanMoonLongitude;
  lambda = dnormalize( lambda, 0., PI2 );
		      
  temp = lambda - meanMoonNode
    + 0.114*TORADIANS*sin(2.*(meanMoonLongitude - meanMoonNode))
      + 0.150*TORADIANS*sin( sunMeanAnomaly ) ;

  /* the sign of the leading term is wrong in Montenbruck, p. 81; cf. Almanac, p. D46 */
  /* Montenbruck's other signs I could not check */
  beta   = 18520. * sin( temp )	  
  - 526.*sin( 2.*meanSunLongitude - meanMoonLongitude - meanMoonNode )
  +  44.*sin( 2.*meanSunLongitude - meanMoonLongitude - meanMoonNode + moonMeanAnomaly )
  -  31.*sin( 2.*meanSunLongitude - meanMoonLongitude - meanMoonNode - moonMeanAnomaly )
  -  23.*sin( 2.*meanSunLongitude - meanMoonLongitude - meanMoonNode + sunMeanAnomaly )
  +  11.*sin( 2.*meanSunLongitude - meanMoonLongitude - meanMoonNode - sunMeanAnomaly )
  -  25.*sin( meanMoonLongitude - meanMoonNode - 2.*moonMeanAnomaly )
  +  21.*sin( meanMoonLongitude - meanMoonNode - moonMeanAnomaly );

  beta /= 3600.;		/* seconds to degrees */
  beta *= TORADIANS;		/* degrees to radians */
  beta = dnormalize( beta, -PI, PI);

  parallax = 3423.
    + 187.*cos( moonMeanAnomaly ) 
    +  10.*cos( 2.*moonMeanAnomaly ) 
    +  34.*cos( 2.*(meanMoonLongitude - meanSunLongitude) - moonMeanAnomaly )
    +  28.*cos( 2.*(meanMoonLongitude - meanSunLongitude) )
    +   3.*cos( 2.*(meanMoonLongitude - meanSunLongitude) + moonMeanAnomaly );

  parallax /= 3600.;
  parallax *= TORADIANS;

  eccliptic->pc_r = 6378134.0 / sin( parallax );
  eccliptic->pc_long = lambda;
  eccliptic->pc_lat = beta;

}


/* ----------------------------------------------------------------
 * tdt_ut returns TDT - UT, about 50-60 seconds in 1990
 * input
 *	UT, in seconds since 1970.0  (UT == universal time == UNIX time )
 * output
 *	TDT - UT, in seconds 
 * side effects 
 * 	none
 * source
 *	almanac, pp. K8-K9
 * accuracy
 *	good only for 1990
 * ----------------------------------------------------------------
 */

double 
tdt_ut( ut )
     double ut;		/* unix time, universal time, seconds since 1970.0 */
{
    return 56.7;
}



/* ----------------------------------------------------------------
 * earthMoonToEarth corrects earth-moon CM position to earth position,
 * inputs
 *	time
 *	ptr to earth-moon position to be corrected 
 * outputs 
 *	none
 * side effects 
 * 	performs the correction 
 * source 
 *	the formula of the almanac, p. E2
 * UNITS on input and output
 *	AU, ASTRONOMICAL UNITS
 * COORDINATE SYSTEM
 *	 supposed to be  mean equator and equinox of J 2000.0
 *	i. e., fk5 standard coord system of referenec
 * ----------------------------------------------------------------
 */

void
earthmoonToEarth( time, position )
     double 	time;		/* unix time, seconds since 1970.0 */
     Coord3d	*position;	/* the position to be corrected; earth-moon CM on input,
				   earth itself on output */
{
    double 	T;		/* time in Julian centuries since JD 2451545.0 */ 
    Coord3d	correction;	/* the correction vector of almanac page E2 */
    double	L;		/* the argument L of p. E2 */
    matrix3d	clipmat;	/* transform from ecliptic of date coords 
				   to ecliptic of J 2000.0 */
    Coord3d	temp;		/* temp vector */

    T = (calendarUnixToJulian( time ) - 2451545.0) / 36525.; 
    L = (218. + 481268. * T) * TORADIANS;
      
    correction.x	= - 0.0000312 * cos(L);
    correction.y	= - 0.0000312 * sin(L);
    correction.z	=   0.0;
    /* the above correction is in coords of mean equinox and ecliptic of date . . . */
    /* change to mean eccliptic and equinox of J 2000.0 */
    cliptoclip2000( time, clipmat );
    matMatVV( clipmat, &correction, &temp );
    GeoScalarMultVector( 1., &temp, &correction ); 

    /* change to mean equator and equinox of J 2000.0 */
    cliptoquat( &correction );

    GeoAddVector( &correction, position, position );

}







/* doubleToDate is in calendar.c */



